# DynamicContent.pm - The Object Class that provides Dynamic Content on the Desktop.
# Created by James A. Pattie, 01/22/2002.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::DynamicContent;
use strict;
use Portal::PortalArgs;
use Portal::Data::Variables;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Args Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

=head1 NAME

DynamicContent - Builds Dynamic Content.

=head1 SYNOPSIS

  use Portal::DynamicContent;
  my $obj = Portal::DynamicContent->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

DynamicContent allows Portal Apps to put content on the Desktop without
having to be running.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(portalSession)

Creates a new instance of the Portal::DynamicContent
object.

requires: portalSession - Portal::Session Object

returns:  object reference

See Portal::PortalArgs(3) for a listing of required arguments.


=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( portalSession => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  my $variableObj = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{variables} = $variableObj;

  $self->{portalSession} = $args{portalSession};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (not defined $self->{portalSession})
  {
    $self->missing("portalSession");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item string display(callingApp, tag, preString, postString, indent, prefName,
                 prefModule)
    Looks up all Dynamic Content entries for the callingApp and tag pair
    and then generates iframe calls wrapped in the preString/postString
    pair.  All content is indented by indent number of spaces.
    The prefName and prefModule values specify where the user Preference
    entry exists which is to be checked to see if this user allows
    Dynamic Content output.  If the preference does not exist, then we
    create it and default to the system default.  If it does exist and
    the value is 0, then we output a "Dynamic Content currently
    disabled" message.

=cut
sub display
{
  my $self = shift;
  my %args = ( callingApp => "", tag => "", preString => "<tr>\n  <td>\n",
               postString => "\n  </td>\n</tr>\n", indent => 2,
               prefModule => "", prefName => "dynamicContent", @_ );
  my $callingApp = $args{callingApp};
  my $tag = $args{tag};
  my $preString = $args{preString};
  my $postString = $args{postString};
  my $indent = $args{indent};
  my $prefModule = $args{prefModule};
  my $prefName = $args{prefName};
  my $result = "";

  # validate the passed in info.
  if (length $callingApp == 0)
  {
    $self->missing("callingApp");
  }
  if (length $tag == 0)
  {
    $self->missing("tag");
  }
  if ($indent !~ /^(\d+)$/)
  {
    $self->invalid("indent", $indent);
  }
  if (length $prefName == 0)
  {
    $self->invalid("prefName", $prefName);
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->postfixError($self->genErrorString("all"));
    return $result;
  }

  # fixup the pre/postString values.
  my $spaces = " " x $indent;
  $preString =~ s/^(.+)$/$spaces$1/mg;
  $postString =~ s/^(.+)$/$spaces$1/mg;
  $spaces = " " x ($indent + 4);  # have to indent the actual content inside of the preString/postString values.

  # verify the callingApp value.
  if ($callingApp ne "Portal")
  {
    my $appObj = $self->{applicationObj}->getApplicationInfo(name => $callingApp);
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return $result;
    }
    if (!defined $appObj)
    {
      $self->error(sprintf($self->langObj->map("callingAppDoesNotExist"), $callingApp));
      return $result;
    }
  }

  # check and see if this user has their preference set.
  my $userId = $self->{portalSession}->{store}->{userObj}->{id};
  my $companyId = $self->{portalSession}->{store}->{companyObj}->{id};
  my $preferenceObj = $self->{authObj}->getOrCreateUserPreference(userId => $userId,
                      app => $callingApp, module => $prefModule, preference => $prefName);
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return $result;
  }
  if (!$preferenceObj->isValid)
  {
    $self->error($preferenceObj->errorMessage);
    return $result;
  }

  # See if the user has the preference set.
  if ($preferenceObj->{value} == 0)
  {
    $result .= $preString . $spaces . "<center>" . $self->langObj->map("DynamicContentDisabled") . "</center>" . $postString;
  }
  else
  {
    # get the Config preference entry or create it if it doesn't exist.
    my $configPrefName = $prefName . "Config";
    my $configPrefObj = $self->{authObj}->getOrCreateUserPreference(userId => $userId, app => $callingApp,
                        module => $prefModule, preference => $configPrefName);
    if ($self->{authObj}->error)
    {
      $self->error($self->{authObj}->errorMessage);
      return $result;
    }
    if (!$configPrefObj->isValid)
    {
      $self->error($configPrefObj->errorMessage);
      return $result;
    }

    # now we need to parse the configPrefObj object
    my $parserObj;
    eval { $parserObj = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
    if ($@)
    {
      $self->error($@);
      return $result;
    }
    my $configObj;
    eval { $configObj = $parserObj->parse(string => $configPrefObj->{value}, module => "DynamicContentConfig", version => "1.0"); };
    if ($@)
    {
      $self->error("Parse failed!<br>\n" . $@);
      return $result;
    }
    my %configSettings = %{$configObj->{settings}};
    my $frameScrolling = $configSettings{frameScrolling};
    my $frameBorder = $configSettings{frameBorder};
    my $frameWidth = $configSettings{frameWidth};
    my $frameHeight = $configSettings{frameHeight};

    # use the portalSession object to determine the company and user info and do the
    # database lookup for any entries that are assigned/created.

    # get all apps in the dynamic_content_tb table.
    my @entries = $self->{applicationObj}->getDynamicContentEntries(companyId => $companyId,
                         userId => $userId, callingApp => $callingApp, tag => $tag, global => 1);

    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return $result;
    }
    my $numApps = 0;
    foreach my $entry (@entries)
    {
      my $app = $entry->{app};
      my $arguments = $entry->{arguments};
      my $url = "";

      # then for each entry, we generate the <iframe> entry.

      if ($app ne "Portal")
      {
        # gather the appInfo
        my $appInfo = $self->{applicationObj}->getApplicationInfo(name => $app);
        if (not defined $appInfo)
        {
          if ($self->{applicationObj}->error)
          {
            $self->error($self->{applicationObj}->errorMessage);
            return $result;
          }
          else
          {
            # The application does not exist!
            $self->error(sprintf($self->langObj->map("invalidApp"), $app));
            return $result;
          }
        }
        else
        {
          if ($appInfo->error)
          {
            $self->error($appInfo->errorMessage);
            return $result;
          }
        }

        # gather the companyAppInfo
        my $companyAppObj = $self->{applicationObj}->getCompanyAppInfo(companyId => $companyId, appId => $appInfo->id);
        if ($self->{applicationObj}->error)
        {
          $self->error($self->{applicationObj}->errorMessage);
          return $result;
        }
        if (!defined $companyAppObj)
        {
          $self->error(sprintf($self->langObj->map("AppNotAssignedToCompany"), $app, $self->{portalSession}->{store}->{companyObj}->{name}));
          return $result;
        }

        # generate the url to execute.
        $url = $self->{methods}->createBaseURL(type => "App", linkType => "cgi", appConfigObj => $companyAppObj);
      }
      else
      {
        $url = $self->{methods}->createBaseURL(type => "Portal", linkType => "cgi");
      }
      # we probably need to urlify the values here!
      $url .= "?app=$app&$arguments&inlineDocument=1";
      my $frameName = $callingApp . $tag . $numApps;

      my $iframePhrase = sprintf($self->langObj->map("IFrameNotSupported"), $app, $arguments);
      my $output = <<"END_OF_CODE";
<iframe src="$url" name="$frameName" scrolling="$frameScrolling" frameborder="$frameBorder" width="$frameWidth" height="$frameHeight">
  [$iframePhrase]
</iframe>
END_OF_CODE

      # now wrap the output in a <tr><td></td></tr> sequence and tack onto the result string.
      $output =~ s/^(.+)$/$spaces$1/mg;
      $result .= $preString . $output . $postString;

      $numApps++;  # keep track of the number of apps we are actually outputing code for.
    }
    if ($numApps == 0)
    {
      $result .= $preString . $spaces . "<center>" . $self->langObj->map("NoDynamicContentAvailable") . "</center>" . $postString;
    }
  }
  return $result;
}

1;

=back

=cut

__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::PortalArgs(3)

=cut
