# Preferences.pm - The Object Class that provides a Preferences Object
# Created by James A. Pattie, 2005-02-25.

# Copyright (c) 2000-2005 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Forms::Preferences;
use strict;
use Portal::AppState;
use Portal::Objects::UserPreferenceObject;
use Portal::Objects::SOAPResult;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.1';

=head1 NAME

Preferences - Object used to build the Preferences Forms Class.

=head1 SYNOPSIS

  use Portal::Forms::Preferences;
  my $obj = Portal::Forms::Preferences->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

  # ask the $obj to process Preferences related forms.

=head1 DESCRIPTION

Preferences is a Preferences Forms class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Forms::Preferences object.
 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item % displayApps(user, url, app, state)

 url - url to specify in the forms action
 app - app that is calling us
 state - state in the app that is calling us
 user - userObj who's preferences are being manipulated.

 Generates a form that allows the user to select the app that
 they want to edit their preferences in.  Only those installed
 apps that have the Preferences object available, will be listed.

 The command that is submitted is: editAppPreferences

 returns %html, where:
   html: hash that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 If an error occurs, we return undef.

=cut
sub displayApps
{
  my $self = shift;
  my %args = ( user => undef, url => "", app => "", state => "", @_ );
  my $user = $args{user};
  my $caller = $self->{userObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};

  # make sure I got valid input.
  if (!defined $user)
  {
    $self->missing("user");
  }
  elsif (!$user->isValid)
  {
    $self->invalid("user", $user->errorMessage);
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  if (defined $user && $user->id != $caller->id)
  {
    # verify the caller is allowed to change the user's info.
    if (!$caller->sysadmin && !$caller->admin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
    }
    elsif (!$caller->sysadmin && $caller->admin && ($user->companyId != $caller->companyId))
    {
      $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $user->{uname})));
    }
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  # display the form.
  my $formName = "selectAppForm";
  my $selectAppPhrase = sprintf($self->{langObj}->map("SelectAppForUser"), $user->get("uname"));
  my $template = <<"END_OF_TEMPLATE";
$selectAppPhrase<br />
#ERROR_MESSAGE#
#STATUS_MESSAGE#
<form #FORMARGS#>
  #FIELD=app#
  #FIELD=state#
  #FIELD=command#
  #FIELD=uname#
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="center">#FIELD=appName#
      </td>
    </tr>
    <tr>
      <td align="center">#FIELD=selectApp#</td>
    </tr>
  </table>
</form>
END_OF_TEMPLATE

  my %apps = $self->applicationObj->getAppsForUser(userId => $user->id);
  if ($self->applicationObj->error)
  {
    $self->error($self->applicationObj->errorMessage);
    return undef;
  }

  my @apps = ();

  # check and see if the Portal has the Preferences object defined.  It better, but it's better to be safe than sorry.
  my $prefObj = "use Portal::Objects::Preferences";
  eval "$prefObj";
  if (!$@)
  {
    push @apps, "Portal";
  }

  # loop over all app groups and find those that have entries.  Then test each entry to see
  # if it has the Preferences object defined and add it to the @apps array if so.
  foreach my $group (sort keys %apps)
  {
    foreach my $appName (sort keys %{$apps{$group}})
    {
      my $prefObj = "use Portal::$appName" . "::Objects::Preferences";
      eval "$prefObj";
      if (!$@)
      {
        push @apps, $appName;
      }
    }
  }

  # pull in the first app and use it's formObj.
  my $prefObj = undef;
  eval "\$prefObj = Portal::" . ($apps[0] ne "Portal" ? $apps[0] . "::" : "") . "Objects::Preferences->new(methods => \$self->methods, authObj => \$self->authObj, applicationObj => \$self->applicationObj, userObj => \$user, appName => \$apps[0], langObj => \$self->langObj);";
  if ($@)
  {
    $self->error($@);
    return undef;
  }
  if ($prefObj->error)
  {
    $self->error($prefObj->errorMessage);
    return undef;
  }

  # now walk the list of returned users and create the %users options structure.
  my $appOptions = $prefObj->formObj->createSelectOptionsFromArrayref(array => \@apps,
    handler => $self->methods->getHandlerSub("Name = Value"));
  if ($prefObj->formObj->error)
  {
    $self->error($prefObj->formObj->errorMessage);
    return undef;
  }

  my %data = ( "selectApp" => { -Type => "submit", -Value => $self->{langObj}->map("SelectApp"), onclick => "if (window.document.$formName.appName.selectedIndex == -1) { alert('" . $self->{langObj}->map("YouMustSelectApp") . "'); return false; } else { return true; }" },
               "appName" => { -Type => "select", -Value => (exists $self->input->{appName} ? $self->input->{appName} : ""), -Label => $self->{langObj}->map("App"), -Options => $appOptions, size => 10 },
               "app" => { -Type => "hidden", -Value => $app },
               "state" => { -Type => "hidden", -Value => $state },
               "command" => { -Type => "hidden", -Value => "editAppPreferences" },
               "uname" => { -Type => "hidden", -Value => $user->get("uname") },
             );

  my %profile = ( required => [], optional => [], constraints => {} );

  # see if we need to display any success/failure messages.
  if (exists $self->{input}->{errorMessage})
  {
    my $string = $self->{methods}->displayMessageStr(type => "error", message => $self->{input}->{errorMessage},
       break => "both");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $template =~ s/#ERROR_MESSAGE#/$string/g;
  }
  else
  {
    $template =~ s/#ERROR_MESSAGE#//g;
  }
  if (exists $self->{input}->{statusMessage})
  {
    my $string = $self->{methods}->displayMessageStr(type => "status", message => $self->{input}->{statusMessage},
       break => "both");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
    $template =~ s/#STATUS_MESSAGE#/$string/g;
  }
  else
  {
    $template =~ s/#STATUS_MESSAGE#//g;
  }

  my %form = $prefObj->formObj->generate(data => \%data,
    template => $template,
    name => $formName,
    action => $url,
    method => "post",
    profile => \%profile);
  if ($prefObj->formObj->error)
  {
    $self->error($prefObj->formObj->errorMessage);
    return undef;
  }

  return %form;
}

=item @ editApp(user, appName, url, app, state)

 url - url to specify in the forms action
 app - app that is calling us
 state - state in the app that is calling us
 user - userObj who's preferences are being manipulated.
 appName - name of the app we are editing preferences in.

 Instantiates an instance of the specified appName's
 Objects::Preferences object which defines the preferences
 that that app allows it's users to edit.

 A form is displayed with the editable preferences displayed
 and this code is called until we have valid input and have
 been able to update/create all the preferences accordingly.

 The command that is submitted is: editAppPreferences

 returns an array with ($result, \%html) as values, where:
   result: 0 = error occured,
           1 = preferences successfully changed,
           2 = html needs to be displayed, we are still processing.
   html: hashref that contains the output to be $doc->print(%html)
     into the callers document for displaying to the user.

 The form is always generated, even if we successfully just updated,
 so that the caller can re-display the form quickly, without having
 to re-call us, etc.

 The resulting form is named: editUserPrefForm

 If a preference doesn't exist for the user, then we check the config
 system to see if a default value has been specified for that
 app, module, preference combo.  If not, then we blow an error and
 quit, else we use the config value as the value to display to the
 user.  The Preference will have _Default added to it when it is
 looked for in the config table.  Thus when defining the config
 entries in the DBSettings.xml file, make sure you use the format
 of <App>_<Module>_<pref>_Default, where <App>, <Module> and <pref>
 are replaced with the values used for defining this preference.

=cut
sub editApp
{
  my $self = shift;
  my %args = ( user => undef, appName => "", url => "", app => "", state => "", @_ );
  my $user = $args{user};
  my $appName = $args{appName};
  my $caller = $self->{userObj};
  my $url = $args{url};
  my $app = $args{app};
  my $state = $args{state};
  my @result = ( 0, {} );

  # make sure I got valid input.
  if (!defined $user)
  {
    $self->missing("user");
  }
  elsif (!$user->isValid)
  {
    $self->invalid("user", $user->errorMessage);
  }
  if (length $appName == 0)
  {
    $self->missing("appName");
  }
  if (length $app == 0)
  {
    $self->missing("app");
  }
  if (length $state == 0)
  {
    $self->missing("state");
  }
  if (defined $user && $user->id != $caller->id)
  {
    # verify the caller is allowed to change the user's info.
    if (!$caller->sysadmin && !$caller->admin)
    {
      $self->invalid("caller", $caller->uname, $self->{langObj}->map("YouAreNotSysOrCompanyAdmin"));
    }
    elsif (!$caller->sysadmin && $caller->admin && ($user->companyId != $caller->companyId))
    {
      $self->invalid("caller", $caller->uname, sprintf($self->{langObj}->map("YouAreCompanyAdminButWrongCompany", $user->{uname})));
    }
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return @result;
  }

  # pull in the appName's Preferences object and use it.
  my $prefName = ($appName ne "Portal" ? $appName . "::" : "");
  eval "use Portal::" . $prefName . "Objects::Preferences;";
  if ($@)
  {
    $self->error($@);
    return @result;
  }
  my $prefObj = undef;
  eval "\$prefObj = Portal::" . $prefName . "Objects::Preferences->new(methods => \$self->methods, authObj => \$self->authObj, applicationObj => \$self->applicationObj, userObj => \$user, appName => \$appName, langObj => \$self->langObj);";
  if ($@)
  {
    $self->error($@);
    return @result;
  }
  if ($prefObj->error)
  {
    $self->error($prefObj->errorMessage);
    return @result;
  }

  my %form = ();  # holds the result of the $formObj->generate() call.
  my $displayForm = 0;
  my $errorStr = "";
  if (!exists $self->input->{formSubmitted})
  {
    $displayForm = 1;
    # loop over all preferences and do the getUserPreferenceInfo() to populate the object.
    foreach my $pref (keys %{$prefObj->data->{main}})
    {
      my $tmpPref = $self->authObj->getUserPreferenceInfo(userId => $user->get("id"),
        app => $appName, module => $prefObj->data->{main}->{$pref}->{-PrefModule},
        preference => $pref);
      if ($self->authObj->error)
      {
        $self->error($self->authObj->errorMessage);
        return @result;
      }
      if (!defined $tmpPref)
      {
        ## lookup the default config value for this instead of blowing up.
        ## If the default config doesn't exist, then we blow an error since it is going to be
        ## standard policy that any preference must have a default config value defined for it.
        my $defaultValue = $self->methods->getConfigDefault(app => $appName, module => $prefObj->{data}->{main}->{$pref}->{-PrefModule}, prefName => $pref . "_Default", portalDB => $self->authObj->portalDB);
        if ($self->methods->error)
        {
          $self->error($self->methods->errorMessage);
          return @result;
        }
        if (!defined $defaultValue)
        {
          $self->error(sprintf($self->langObj->map("DefaultForPreferenceDoesNotExist"), $pref, $appName, $prefObj->{data}->{main}->{$pref}->{-PrefModule}, $user->get("id")));
          return @result;
        }
        else
        {
          # we got the default value, so now we just need to create the preference using it.
          $tmpPref = Portal::Objects::UserPreferenceObject->new(id => $user->get("id"),
                        app => $appName, module => $prefObj->data->{main}->{$pref}->{-PrefModule}, name => $pref,
                        value => $defaultValue, langObj => $self->{langObj});
        }
      }
      if ($tmpPref->error)
      {
        $self->error($tmpPref->errorMessage);
        return @result;
      }
      # this populates the main and edit forms.
      $prefObj->set($pref => $tmpPref->get("value"));
    }
  }
  else
  {
    # validate the input and update/create the preferences if everything ok.
    my $result = $prefObj->validateForm(input => $self->input, form => "edit");
    if ($prefObj->error)
    {
      $self->error($prefObj->errorMessage);
      return @result;
    }
    if ($result)
    {
      # we appear to have valid data.  Now to loop over all the preference entries
      # and call validateEntry() on them to give the Preferences object one last
      # chance to do any extra validation and to fixup the data as needed and
      # put it back into data->{main} for me.
      foreach my $pref (keys %{$prefObj->data->{main}})
      {
        if (!$prefObj->validateEntry($pref))
        {
          $displayForm = 1;
        }
      }
      if (!$displayForm)
      {
        # we made it this far, now we get to update/create the users preferences.
        foreach my $pref (keys %{$prefObj->data->{main}})
        {
          my $tmpPrefObj = Portal::Objects::UserPreferenceObject->new(id => $user->get("id"),
                        app => $appName, module => $prefObj->data->{main}->{$pref}->{-PrefModule}, name => $pref,
                        value => $prefObj->get($pref), langObj => $self->{langObj});
          if ($tmpPrefObj->error)
          {
            $self->error($tmpPrefObj->errorMessage);
            return @result;
          }
          my $result = $self->authObj->updateUserPreference(userPreferenceObj => $tmpPrefObj);
          if ($self->authObj->error)
          {
            $self->error($self->authObj->errorMessage);
            return @result;
          }
          if ($result == -1)
          { # have to create it
            $result = $self->authObj->createUserPreference(userPreferenceObj => $tmpPrefObj);
            if ($self->authObj->error)
            {
              $self->error($self->authObj->errorMessage);
              return @result;
            }
            if ($result == -1)
            { # already existed!  don't know how!
              $self->error(sprintf($self->langObj->map("createUserPreferenceNowExists"), $tmpPrefObj->print));
              return @result;
            }
            elsif ($result == -2)
            { # app doesn't exist
              $self->error(sprintf($self->langObj->map("createUserPreferenceAppNotExist"), $tmpPrefObj->print));
              return @result;
            }
            elsif ($result == 1)
            { # we are ok.
            }
          }
          elsif ($result == -2)
          { # app doesn't exist
            $self->error(sprintf($self->langObj->map("updateUserPreferenceAppNotExist"), $tmpPrefObj->print));
            return @result;
          }
          elsif ($result == 1)
          { # we are ok.
          }
        }

        # signal that we successfully updated the user preferences.
        $result[0] = 1;
        $displayForm = 1;
      }
    }
    else
    {
      $displayForm = 1;
    }
  }

  if ($displayForm)
  {
    # populate the appName value.
    $prefObj->data->{edit}->{appName}->{-Value} = $appName;

    # now loop over all the preferences and call setupEntry() so that they can properly populate the edit form.
    foreach my $pref (keys %{$prefObj->data->{main}})
    {
      $prefObj->setupEntry($pref);
      if ($prefObj->error)
      {
        $self->error($prefObj->errorMessage);
        return @result;
      }
    }

    # now generate the form.
    %form = $prefObj->printForm(form => "edit", action => $url,
      app => $app, state => $state);
    if ($prefObj->error)
    {
      $self->error($prefObj->errorMessage);
      return @result;
    }

    $result[0] = 2 if ($result[0] == 0);  # signal we have output to display to the user.
    $result[1] = \%form;
  }

  return @result;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::AppState(3)

=cut
