# Login.pm - The Object Class that provides a Login Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Login;
use strict;
use Portal::PortalArgs;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::PortalArgs Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.12';

# global variables.
my %commands = ( display => "Login Screen", authenticate => "Authenticate Login Attempt" );

=head1 NAME

Login - Object used to build a Login Object Class.

=head1 SYNOPSIS

  use Portal::Login;
  my $obj = Portal::Login->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Login is a Login class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::Login object.
 See Portal::PortalArgs(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject run(state, command)

 Pulls the specified state file into existance and calls
 run(command).

 returns: The HTMLObject document to display or undef if an error.

=cut

sub run
{
  my $self = shift;
  my %args = ( command => "", @_ );
  my $command = $args{command};
  my $doc;

  if (!exists $commands{$command})
  {
    $self->error(sprintf($self->langObj->map("commandNotKnown"), $self->formEncodeString($command)));
    return undef;
  }

  $doc = HTMLObject::Normal->new;
  if ($command eq "display")
  {
    $doc = $self->displayLoginForm(doc => $doc, input => $self->{input});
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
  }
  elsif ($command eq "authenticate")
  {
    my $uname = $self->{input}->{uname};
    my $passwd = $self->{input}->{passwd};
    if (!exists $self->{input}->{uname})
    {
      $doc = $self->displayLoginForm(doc => $doc, error => $self->langObj->map("unameNeeded"));
      return $doc;
    }
    if (length $uname == 0)
    {
      $doc = $self->displayLoginForm(doc => $doc, error => $self->langObj->map("unameNeeded"));
      return $doc;
    }
    if (!exists $self->{input}->{passwd})
    {
      $doc = $self->displayLoginForm(doc => $doc, error => $self->langObj->map("passwdNeeded"));
      return $doc;
    }
    if (length $passwd < 6)
    {
      $doc = $self->displayLoginForm(doc => $doc, error => $self->langObj->map("passwdLengthError"), uname => $uname);
      return $doc;
    }

    my $result = $self->{authObj}->isUserValid(uname => $uname, password => $passwd);
    if ($self->{authObj}->error)
    {
      $self->error($self->{authObj}->errorMessage);
      return undef;
    }
    if ($result != 1) # somethings wrong!
    {
      if ($result == 0)  # invalid password
      {
        $doc = $self->displayLoginForm(doc => $doc, error => $self->langObj->map("invalidPassword"), uname => $uname);
        return $doc;
      }
      elsif ($result == -1)  # user not active
      {
        $doc = $self->displayLoginForm(doc => $doc, error => sprintf($self->langObj->map("userNotActive"), $uname));
        return $doc;
      }
      elsif ($result == -2)  # company not active
      {
        $doc = $self->displayLoginForm(doc => $doc, error => sprintf($self->langObj->map("companyNotActive"), $uname));
        return $doc;
      }
      elsif ($result == -3)  # company doesn't exist
      {
        $doc = $self->displayLoginForm(doc => $doc, error => sprintf($self->langObj->map("companyNotFound"), $uname));
        return $doc;
      }
      elsif ($result == -4)  # user not found
      {
        $doc = $self->displayLoginForm(doc => $doc, error => sprintf($self->langObj->map("userNotFound"), $uname));
        return $doc;
      }
      elsif ($result == -5)  # user locked out
      {
        $doc = $self->displayLoginForm(doc => $doc, error => sprintf($self->langObj->map("userLockedOut"), $uname));
        return $doc;
      }
    }
    else  # generate the Desktop.
    {
      my $userId = $self->{authObj}->getUserId(uname => $uname);
      if ($self->{authObj}->error)
      {
        $self->error($self->{authObj}->errorMessage);
        return undef;
      }

      my $logEntry = Portal::Objects::LogEntry->new(action => 0, ipAddress => (exists $ENV{REMOTE_ADDR} ? $ENV{REMOTE_ADDR} : '127.0.0.1'), userId => $userId, langObj => $self->{langObj});
      if ($logEntry->error)
      {
        $self->error($logEntry->errorMessage);
        return undef;
      }

      # log the login event.
      $self->{logObj}->newEntry(logEntry => $logEntry);
      if ($self->{logObj}->error)
      {
        $self->error($self->{logObj}->errorMessage);
        return undef;
      }

      # pull the Desktop into existance
      use Portal::Desktop;
      my $desktop = undef;
      my $arguments = $self->arguments();
      eval "\$desktop = Portal::Desktop->new($arguments, startSession => 1);";
      if ($@)
      {
        my $string = $self->langObj->map("stateEvalFailed");
        $self->error(sprintf($string, "Portal::Desktop Object", $@));
        return $doc;
      }
      if ($desktop->error)
      {
        $self->error($desktop->errorMessage . "\n<br>arguments='$arguments'");
        return undef;
      }

      # call run (command => display);
      eval {
      $doc = $desktop->run(command => "display");
      };
      if ($@)
      {
        my $string = $self->langObj->map("stateEvalFailed");
        $self->error(sprintf($string, "Portal::Desktop Run", $@));
        $self->error("startSession = '" . $desktop->{startSession} . "'");
        return $doc;
      }
      if ($desktop->error)
      {
        $self->error($desktop->errorMessage);
        return undef;
      }
    }
  }

  return $doc;
}

sub displayLoginForm
{
  my $self = shift;
  my %args = ( error => "", uname => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $uname = $args{uname};
  my $input = $args{input};
  my $message = "";

  my $url = $self->{methods}->createBaseURL(type => "Portal", linkType => "cgi");

  # get the users colorScheme to work with.
  my $colorScheme = $self->{applicationObj}->getCurrentColorScheme(userId => "-1");
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    return $doc;
  }
  if (!defined $colorScheme)
  {
    $self->error("No valid colorScheme returned!<br>\n");
    return $doc;
  }

  my $login = $self->langObj->map("login");
  my $userName = $self->langObj->map("userName");
  my $passwd = $self->langObj->map("password");
  my $portalPhrase = $self->langObj->map("portal");
  my $hostedBy = $self->langObj->map("hostedBy");
  my $cookiesJavascript = $self->langObj->map("cookiesJavascript");
  my $copyright = $self->langObj->map("copyright");
  my $pcxName = $self->langObj->map("pcxName");
  my $license = $self->langObj->map("license");
  my $errorPhrase = $self->langObj->map("error");
  my $screen = $self->langObj->map("screen");
  my $passThruLogin = $self->langObj->map("passThruLogin");
  my $app = "Portal";
  my $state = "Login";
  my $command = "authenticate";
  my $extraTags = "";
  my $unameInputStr = "<input type=\"text\" name=\"uname\" value=\"$uname\">";
  my $hostingURL = $self->{configObj}->hostingURL;
  my $hostingAltTag = $self->{configObj}->hostingAltTag;
  my $hostingLogoUrl = $self->{configObj}->hostingLogoUrl;

  if (exists $input->{client})
  {
    $app = $input->{app};
    $state = $input->{state};
    $userName = $self->langObj->map("clientUserName");
    $unameInputStr = "<span class=\"login\">$uname</span>";
  }

  # make sure that any tags the user passed in are preserved.
  foreach my $tag (keys %{$input})
  {
    my $value = $input->{$tag};
    if ($tag =~ /^(appToRun|appToRunArgs)$/)
    {
      $extraTags .= "  <input type=\"hidden\" name=\"$tag\" value=\"$value\">\n";
    }
  }

  if (exists $input->{appToRun} && exists $input->{appToRunArgs})
  {
    $message = sprintf($passThruLogin, $input->{appToRun});
  }

  my $portalVersionPhrase = sprintf("$portalPhrase", $Portal::VERSION);

  # create the needed css entries
  my %cssEntries = $self->{methods}->generateColorSchemeCSS(colorScheme => $colorScheme);
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  $doc->print(%cssEntries);

  $doc->setTitle("$portalVersionPhrase - $login $screen");
  $doc->setFocus("body");
  $doc->print(<<"END_OF_CODE");
<center>
<h2>$portalVersionPhrase - $login</h2>
$hostedBy
<br><a href="$hostingURL" title="$hostingAltTag" target="host"><img src="$hostingLogoUrl" alt="$hostingAltTag" border="0"></a>
<br>
<br>
<!-- <font size="-1">($cookiesJavascript)</font>
<br> -->
<form method="post" action="$url" name="frmLogin">
$extraTags
  <input type="hidden" name="app" value="$app">
  <input type="hidden" name="state" value="$state">
  <input type="hidden" name="command" value="$command">
  <table border="0" cellpadding="0" cellspacing="0" class="login">
    <tr>
      <td align="right">$userName&nbsp;:&nbsp;</td>
      <td align="left">$unameInputStr</td>
    </tr>
    <tr>
      <td align="right">$passwd&nbsp;:&nbsp;</td>
      <td align="left"><input type="password" name="passwd" value=""></td>
    </tr>
  </table>
  <input type=submit name="login" value="$login">
</form>
END_OF_CODE
  if (length $error > 0)
  {
    # output an error Message
    $doc = $self->{methods}->displayMessage(doc => $doc, type => "error", message => $error);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }
  if (length $message > 0)
  {
    # output an information Message
    $doc = $self->{methods}->displayMessage(doc => $doc, type => "status", message => $message);
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }
  $doc->print(<<"END_OF_CODE");
  <br>
  <br>
  <font size="-1">$copyright <a href="http://www.pcxperience.com/" target="pcx">$pcxName</a>
  <br>
  $license</font>
</center>
END_OF_CODE

  # generate the code to have either the uname or passwd field selected by default.
  my $onLoadCode = "";
  if (length $uname > 0)
  {  # select the password box
    $onLoadCode = "document.frmLogin.passwd.focus();\n";
  }
  else
  {  # select the username box
    $onLoadCode = "document.frmLogin.uname.focus();\n";
  }

  # Uncomment the following line when we can make this happen on all browsers (Version 4 and 5)
  #$onLoadCode .= "window.open('','_top','menubar=0,location=0,toolbar=0,personalbar=0,status=1,scrollbars=1,resizable=1,close=0');\n";

  $doc->setOnload(code => $onLoadCode);

  # set the JavaScript Error Handler Email Address
  $doc->setJavascriptErrorHandlerEmail(email => $self->{configObj}->{emailAddress});
  # set the JavaScript Error Handler Program Name and Version info
  $doc->setJavascriptErrorHandlerPrgName(name => "Portal");
  $doc->setJavascriptErrorHandlerPrgVersion(version => $VERSION);

  return $doc;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::PortalArgs(3)

=cut
