# Methods.pm - The Object Class that provides common methods
# Created by James A. Pattie, 10/15/2002.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Methods;
use Portal::Base;
use DBIWrapper;
use HTMLObject::Base;
use Portal::Session;
use Portal::SessionHandler;
use Portal::Objects::MenuItem;
use Time::Local;
use POSIX qw(strftime);
use Date::Manip qw(ParseDate UnixDate DateCalc Date_ConvTZ);
use Portal::Log;
use Portal::Objects::LogEntry;
use Portal::Objects::MenuItem;
use Portal::Auth;
use Portal::Application;
use strict;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

=head1 NAME

Methods - Object used to build a common methods Object Class.

=head1 SYNOPSIS

  use Portal::Methods;
  my $obj = Portal::Methods->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Methods is a class that provides some common Portal methods.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Methods module.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = (  @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item scalar displayMenu(itemsLeft, itemsCenter, itemsRight, orientation, indent)

 requires: itemsLeft, indent
 optional: itemsCenter, itemsRight, orientation
 returns:  string with menu created in it.
   indent is the number of spaces to indent in.
   itemsLeft, itemsCenter and itemsRight are arrays of the items you
          want displayed.  Each item is an instance of the MenuItem
          Object.  itemsCenter can only be used if all three arrays
          are specified.
   orientation is the direction the menu should be displayed in.  The
          available values are (horizontal, vertical).  We default to
          horizontal if nothing specified.  If vertical, then only
          the itemsLeft array will be allowed.

=cut
sub displayMenu
{
  my $self = shift;
  my %args = ( itemsLeft => undef, itemsCenter => undef,
               itemsRight => undef, orientation => "horizontal", indent => "0", @_ );
  my $indent = $args{indent};
  my $itemsLeft = $args{itemsLeft};
  my $itemsCenter = $args{itemsCenter};
  my $itemsRight = $args{itemsRight};
  my $orientation = $args{orientation};

  if ($indent !~ /^\d+$/)
  {
    $self->invalid("indent", $indent);
  }
  if (!defined $itemsLeft)
  {
    $self->missing("itemsLeft");
  }
  elsif (scalar @{$itemsLeft} == 0)
  {
    $self->invalid("itemsLeft", "must have at least 1 entry!");
  }
  if (defined $itemsCenter && !defined $itemsRight)
  {
    $self->invalid("itemsCenter", "can not be defined without itemsRight!");
  }
  if (defined $itemsCenter && scalar @{$itemsCenter} == 0)
  {
    $self->invalid("itemsCenter", "must have at least 1 entry!");
  }
  if (defined $itemsRight && scalar @{$itemsRight} == 0)
  {
    $self->invalid("itemsRight", "must have at least 1 entry!");
  }
  if ($orientation !~ /^(horizontal|vertical)$/)
  {
    $self->invalid("orientation", $orientation);
  }
  if ($orientation eq "vertical" && defined $itemsRight)
  {
    $self->invalid("You can not define itemsRight/itemsCenter", "when orientation = vertical!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return "";
  }

  # we are generating a table to fit within whatever layout the user builds
  # around us.
  my $indentString = " " x $indent;

  my $string = <<"END_OF_CODE";
<table border="0" cellpadding="0" cellspacing="0" class="menu" width="100%">
END_OF_CODE

  if ($orientation eq "horizontal")
  {
    # we are in a single <tr></tr> pair.
    $string .= "  <tr>\n";

    # do we have all 3 arrays?
    if (defined $itemsRight && defined $itemsCenter)
    {
      $string .= "    <td width=\"34%\" align=\"left\">\n$indentString";
      foreach my $entry (@{$itemsLeft})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
      $string .= "    <td width=\"33%\" align=\"center\">\n$indentString";
      foreach my $entry (@{$itemsCenter})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
      $string .= "    <td width=\"33%\" align=\"right\">\n$indentString";
      foreach my $entry (@{$itemsRight})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
    }
    # do we have left and right?
    elsif (defined $itemsRight && !defined $itemsCenter)
    {
      $string .= "    <td width=\"50%\" align=\"left\">\n$indentString";
      foreach my $entry (@{$itemsLeft})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
      $string .= "    <td width=\"50%\" align=\"right\">\n$indentString";
      foreach my $entry (@{$itemsRight})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
    }
    # do we only have left?
    else
    {
      $string .= "    <td width=\"100%\" align=\"left\">\n$indentString";
      foreach my $entry (@{$itemsLeft})
      {
        my $result = $entry->print(orientation => $orientation, indent => 6);
        if ($entry->error)
        {
          $self->error($entry->errorMessage);
          return "";
        }
        $string .= $result;
      }
      $string .= "\n    </td>\n";
    }
    $string .= "  </tr>\n";
  }
  else
  {
    # each menuitem is in it's own <tr></tr> pair.
    foreach my $entry (@{$itemsLeft})
    {
      $string .= "  <tr>\n    <td width=\"100%\" align=\"left\">\n$indentString";
      my $result = $entry->print(orientation => $orientation, indent => 6);
      if ($entry->error)
      {
        $self->error($entry->errorMessage);
        return "";
      }
      $string .= $result;
      $string .= "\n    </td>\n  </tr>\n";
    }
  }

  $string .= "</table>";

  # now indent the output.
  $string =~ s/^(.*)$/$indentString$1/mg;

  return $string;
}

=item -1,0,1 sortCSS($a, $b)

 Tries to make sure that the css a tags come out in the correct
 order for :link, :active, :visited, :hover

=cut
sub sortCSS
{
  if ($a =~ /a(\.[^\s]+)?:link/ && $b =~ /a(\.[^\s]+)?(:hover|:active)/)
  {
    -1;  # less than
  }
  elsif ($a =~ /a(\.[^\s]+)?(:hover|:active)/ && $b =~ /a(\.[^\s]+)?:link/)
  {
    1;   # greater than
  }
  elsif ($a =~ /a(\.[^\s]+)?:hover/ && $b =~ /a(\.[^\s]+)?:visited/)
  {
    1;   # greater than
  }
  elsif ($a =~ /a(\.[^\s]+)?:visited/ && $b =~ /a(\.[^\s]+)?:hover/)
  {
    -1;  # less than
  }
  else
  {
    $a cmp $b;
  }
}

=item % generateColorSchemeCSS(colorScheme, appName)

  requires: colorScheme, appName
  returns: hash of CSS entries that can be $doc->print'ed.
  colorScheme is the colorScheme Object that represents the users
        current colorScheme.
  appName is the name of the app we are generating CSS entries for.
        If none specified, we default to Portal.

=cut
sub generateColorSchemeCSS
{
  my $self = shift;
  my %args = ( colorScheme => undef, appName => "Portal", @_ );
  my $colorScheme = $args{colorScheme};
  my $appName = $args{appName};
  my %generatedEntries = ();  # keeps track of nonPortal entries created in the Portal loop.
  my %entries = ( css => [] );  # generated CSS entries to return.

  if (!defined $colorScheme)
  {
    $self->missing("colorScheme");
  }
  if ($appName !~ /^(Portal|.+)$/)
  {
    $self->invalid("appName", $appName);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return %entries;
  }

  my @PortalCSSEntries = sort sortCSS sort grep { /^Portal:.*$/ } @{$colorScheme->{'_names_'}};

  my @SecondaryCSSEntries = ();
  if ($appName ne "Portal")
  {
    @SecondaryCSSEntries = sort sortCSS sort grep { /^$appName:.*$/ } @{$colorScheme->{'_names_'}};
  }

  # we have to loop over all CSS entries for the specified app (or Portal first and then
  # the specified app) and generate the entries as we encounter them.  Try to make sure we
  # order Link, Active, Visited, Hover in that order for a tags and bare, Focus, Hover,
  # Hover+Focus for input, select, textarea and button tags (where bare means no modifier).

  # first loop is for Portal: entries

  foreach my $entry (@PortalCSSEntries)
  {
    # break it down and figure out what type of entry we are working with.
    $entry =~ /^(Portal:)(.+)$/;

    # build up the CSS tag name.
    my $tagName = $2;

    # build up the override entry name.
    my $overrideCSS = $appName . ":$tagName";

    # always generate the Portal entry first
    my $cssEntry = $tagName . " { $colorScheme->{$entry} }\n";
    push @{$entries{css}}, $cssEntry;

    # see if the App has an override entry and generate it.
    if ($appName ne "Portal" && exists $colorScheme->{$overrideCSS})
    {
      my $cssEntry = $tagName . " { $colorScheme->{$overrideCSS} }\n";
      push @{$entries{css}}, $cssEntry;

      # keep track that we used the app version instead of the Portal version.
      $generatedEntries{$overrideCSS} = 1;
    }
  }

  if ($appName ne "Portal")
  {
    # second loop is for appName specific entries
    foreach my $entry (@SecondaryCSSEntries)
    {
      next if (exists $generatedEntries{$entry});  # this entry has already been generated so skip.

      # break it down and figure out what type of entry we are working with.
      $entry =~ /^($appName:)(.+)$/;

      # build up the CSS tag name.
      my $tagName = $2;

      my $cssEntry = $tagName . " { $colorScheme->{$entry} }\n";
      push @{$entries{css}}, $cssEntry;
    }
  }

  return %entries;
}

=item HTMLObject displayMessage(doc, type, message, langObj, indent, break)

 requires: doc, type, message, indent, break
 optional: langObj
 returns: doc with message table created in it.

 doc is an HTMLObject object.
 indent is the number of spaces to indent in.
 type = 'error' or 'status' and defines the type of message.
 message is the error or status message to display.
 break = above, before, below, after, both, none.  specifies where the
        <br /> is to go.
 langObj is the Portal::Language object which provides the phrases
        to display in the users language.  If not specified, then
        we use the Portal::Methods langObj instance.

 Notes: type='error' requires phrase 'error' to be defined.
        type='status' requires phrase 'status'.

=cut
sub displayMessage
{
  my $self = shift;
  my %args = ( doc => undef, type => "error", message => "", indent => "2", break => "above", @_ );
  my $doc = $args{doc};
  my $type = $args{type};
  my $message = $args{message};
  my $langObj = (exists $args{langObj} ? $args{langObj} : $self->{langObj});
  my $indent = $args{indent};
  my $break = $args{break};

  if (!defined $doc)
  {
    $self->missing("doc");
  }
  if (!defined $langObj)
  {
    $self->missing("langObj");
  }
  if ($type !~ /^(error|status)$/)
  {
    $self->invalid("type", $type);
  }
  if (length $message == 0)
  {
    $self->invalid("message", $message);
  }
  if ($indent !~ /^\d+$/)
  {
    $self->invalid("indent", $indent);
  }
  if ($break !~ /^(above|before|below|after|both|none)$/)
  {
    $self->invalid("break", $break);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $doc;
  }

  my $phrase = $langObj->map("$type");
  my $cssText = $type . "Text";
  my $string = "";

  # store current document focus
  my $oldFocus = $doc->getFocus;

  # set focus to body section
  $doc->setFocus("body");

  if ($break =~ /^(above|before|both)$/)
  {
    $string .= $doc->br . "\n";
  }
  $string .= $doc->table(border => 0, cellpadding => 2, cellspacing => 2, class => $type, -content =>
    $doc->tr(-content =>
    $doc->td(align => "right", valign => "top", -content => "$phrase:") . $doc->td(align => "left", valign => "top", -content =>
    $doc->span(class => $cssText, -content => $message))));

  if ($break =~ /^(below|after|both)$/)
  {
    $string .= $doc->br . "\n";
  }

  # now indent the output.
  my $indentString = " " x $indent;
  $string =~ s/^(.*)$/$indentString$1/mg;

  # now print the message into the document
  $doc->print($string);

  # restore previous document focus
  $doc->setFocus($oldFocus);

  return $doc;
}

=item scalar displayMessageStr(doc, type, message, langObj, indent, break)

 requires: doc, type, message, indent, break
 optional: langObj
 returns: string with message table created in it.

 doc is an HTMLObject object.  If not specified, an HTMLObject::Base
   instance will be instantiated and used.
 indent is the number of spaces to indent in.
 type = 'error' or 'status' and defines the type of message.
 message is the error or status message to display.
 break = above, before, below, after, both, none.  specifies where
        the <br /> is to go.
 langObj is the Portal::Language object which provides the phrases
        to display in the users language.  If not specified, then
        we use the Portal::Methods langObj instance.

 Notes: type='error' requires phrase 'error' to be defined.
        type='status' requires phrase 'status'.

=cut
sub displayMessageStr
{
  my $self = shift;
  my %args = ( doc => undef, type => "error", message => "", indent => "2", break => "above", @_ );
  my $doc = $args{doc};
  my $type = $args{type};
  my $message = $args{message};
  my $langObj = (exists $args{langObj} ? $args{langObj} : $self->{langObj});
  my $indent = $args{indent};
  my $break = $args{break};
  my $string = "";

  if (!defined $doc)
  {
    $doc = HTMLObject::Base->new;
  }
  if (!defined $langObj)
  {
    $self->missing("langObj");
  }
  if ($type !~ /^(error|status)$/)
  {
    $self->invalid("type", $type);
  }
  if ($indent !~ /^\d+$/)
  {
    $self->invalid("indent", $indent);
  }
  if ($break !~ /^(above|before|below|after|both|none)$/)
  {
    $self->invalid("break", $break);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $string;
  }

  if (length $message == 0)
  {
    # short-circuit out if we aren't given anything to display, but don't treat it as an error.
    return $string;
  }

  my $phrase = $langObj->map("$type");
  my $cssText = $type . "Text";

  # store current document focus
  my $oldFocus = $doc->getFocus;

  if ($break =~ /^(above|before|both)$/)
  {
    $string .= $doc->br;
  }
  $string .= $doc->table(border => 0, cellpadding => 2, cellspacing => 2, class => $type, -content =>
    $doc->tr(-content =>
    $doc->td(align => "right", valign => "top", -content => "$phrase:") . $doc->td(align => "left", valign => "top", -content =>
    $doc->span(class => $cssText, -content => $message))));

  if ($break =~ /^(below|after|both)$/)
  {
    $string .= $doc->br;
  }

  # now indent the output.
  my $indentString = " " x $indent;
  $string =~ s/^(.+)$/$indentString$1/mg;

  return $string;
}

=item HTMLObject displayJSHelper(doc, type, langObj)

 requires: doc, type
 optional: langObj
 returns: doc with specified JavaScript function defined.
 doc is an HTMLObject::Normal object.
 type = 'logOut', 'closeApp', 'closeWindow', 'launchApp' or 'help'
        and defines the type of JavaScript function to create.
        This can be a comma seperate list of options, if you want
        multiple generated at once.
 langObj is the Portal::Language object which provides the phrases
        to display in the users language.  If not specified, then
        we use the Portal::Methods langObj instance.

 Notes: type='logOut' requires phrase 'logoutPortal' to be defined.
        type='closeApp' requires phrase 'closeMessage'.

=cut
sub displayJSHelper
{
  my $self = shift;
  my %args = ( doc => undef, type => "logOut", @_ );
  my $doc = $args{doc};
  my $type = $args{type};
  my $langObj = (exists $args{langObj} ? $args{langObj} : $self->{langObj});
  my @types = split /\s*,\s*/, $type;

  if (!defined $doc)
  {
    $self->missing("doc");
  }
  foreach $type (@types)
  {
    if ($type !~ /^(logOut|closeApp|closeWindow|launchApp|help)$/)
    {
      $self->invalid("type", $type);
    }
  }
  if (!defined $langObj)
  {
    $self->missing("langObj");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $doc;
  }

  # store current document focus
  my $oldFocus = $doc->getFocus;

  # set focus to JavaScript section
  $doc->setFocus("javascript");

  foreach $type (@types)
  {
    if ($type =~ /^(logOut|closeApp)$/)
    {
      my $closePhrase = ($type eq "logOut" ? $langObj->map("logoutPortal") : $langObj->map("closeMessage"));
      $doc->print(<<"END_OF_CODE");

function validateClose()
{
  return window.confirm("$closePhrase");
}

END_OF_CODE
    }
    elsif ($type eq "closeWindow")
    {
      $doc->print(<<"END_OF_CODE");

function closeWindow(wname)
{
  var w = open("", wname);
  if (!w.closed)
  {
    w.close();
  }
}

END_OF_CODE
    }
    elsif ($type eq "launchApp")
    {
      $doc->print(<<"END_OF_CODE");

function launchApp(appName, url, height, width)
{
  var w = window.open(url, appName, "height="+height+",width="+width+",resizable,scrollbars,status,close=no");
  return w;  // not sure if this will be used at all.
}

END_OF_CODE
    }
    elsif ($type eq "help")
    {
      $doc->print(<<"END_OF_CODE");

function showHelp(appName, url, height, width)
{
  var w = window.open(url, appName, "height="+height+",width="+width+",resizable,scrollbars,status");
}

END_OF_CODE
    }
  }

  # restore previous document focus
  $doc->setFocus($oldFocus);

  return $doc;
}

=item % generatePickerCode()

 This method has been removed from Portal::Methods.

 Replace your call to $self->methods->generatePickerCode() with the
 following code snippet where %args is all the arguments you
 were passing to the Portal::Methods version.

 # now call the HTMLObject::Widgets->generatePickerCode() method.
 use HTMLObject::Widgets;
 my $widgets = HTMLObject::Widgets->new();

 my %result = $widgets->generatePickerCode(%args, baseUrl => $self->baseUrl);
 if ($widgets->error)
 {
   $self->error($widgets->errorMessage());
   return undef;
 }
 $doc->print(%result);


 See the HTMLObject::Widgets man page for the args to pass the
 generatePickerCode() method.

 $self->baseUrl is only valid if you are in an object derived from
 Portal::AppState and that object called $self->buildUrls.

=cut

=item scalar includeCoreJSFiles(doc)

 Outputs the necessary JS Include statements to include the
 core external JavaScript files that the Portal provides.

 requires: doc - HTMLObject::Normal instance
 returns: doc

=cut

sub includeCoreJSFiles
{
  my $self = shift;
  my %args = ( doc => undef, @_ );
  my $doc = $args{doc};

  if (!defined $doc)
  {
    $self->missing("doc");
    $self->error($self->genErrorString("all"));
    return $doc;
  }

  my $jsUrl = $self->createBaseURL(type => "Portal", linkType => "base");
  if ($self->error)
  {
    $self->error($self->errorMessage());
    return $doc;
  }
  $doc->setJavascriptInclude(file => $jsUrl . "/htmlobject/js/form_methods.js");

  return $doc;
}

=item scalar createBaseURL(type, linkType, appConfigObj, appName)

  requires: type, linkType
  optional: appConfigObj, appName
  returns:  base url suitable for use in forms or that can have
        ?args=x&argv=y, etc. tacked on for use in a link.
        If linkType != cgi, then the url does end with a /.
        If type = 'App' then the AppName is strategically placed
          in the url as appropriate.
  type = 'Portal' or 'App', where we default to 'Portal'
  linkType = 'cgi', 'cgibin', 'image', 'js', 'db_dump', 'install',
          'css', 'soap', 'soapcgi' or 'base'.  We default to 'cgi'.
  appConfigObj is the App's companyAppObj if type = 'App'.
  appName = name of App when type = 'App' and you are not calling
          from within the App itself.  This is used mainly for the
          Portal itself in relation to the Desktop images.

  linkType = 'cgi' points to index.cgi
  linkType = 'cgibin' returns the cgi-bin directory.
  linkType = 'soap' returns the soap directory.
  linkType = 'soapcgi' points to the app's soap cgi script.
    If type = 'Portal', then this is portal.cgi,
    else if type = 'App', then this is the lowercase app name.cgi.
    Ex: appName = 'Accounting', then we would point to accounting.cgi.

  linkType = 'base' returns the url minus the portal specific part.

  Example usage:
  $methods->createBaseURL(type => "App", linkType => "cgi",
     appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});

  Would return "http://www.test.com/portal/cgi-bin/index.cgi"
  assuming that the App was installed on www.test.com.

  $methods->createBaseURL(type => "App", linkType => "js",
     appConfigObj => $self->{sessionObj}->{store}->{companyAppObj});

  Would return "http://www.test.com/portal/js/Accounting/" assuming
  that the App was installed on www.test.com and was Accounting
  (xiwa).

  $methods->createBaseURL(type => "Portal", linkType => "base");

  Would return "http://www.test.com" assuming that the Portal was
  installed on www.test.com was using http, not https.

=cut
sub createBaseURL
{
  my $self = shift;
  my %args = ( type => "Portal", linkType => "cgi", appConfigObj => undef, appName => "", @_ );
  my $type = $args{type};
  my $linkType = $args{linkType};
  my $configObj = undef;
  my $appConfigObj = $args{appConfigObj};
  my $url = "";
  my $appName = $args{appName};

  eval { $configObj = Portal::Data::Config->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->error("Instantiating Portal configObj failed!<br>\n$@");
    return $url;
  }
  if ($configObj->error)
  {
    $self->error($configObj->errorMessage);
    return $url;
  }

  if ($type !~ /^(Portal|App)$/)
  {
    $self->invalid("type", $type);
  }
  if ($linkType !~ /^(cgi|cgibin|image|js|css|db_dump|install|soap|soapcgi|base)$/)
  {
    $self->invalid("linkType", $linkType);
  }
  if ($type eq "App" && !defined $appConfigObj)
  {
    $self->missing("appConfigObj");
  }
  if ($type eq "App" && $linkType eq "base")
  {
    $self->invalid("linkType", $linkType, "type = 'App' does not support this linktype");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $url;
  }

  if ($type eq "Portal")
  {
    $url = $configObj->{httpType} . "://" . $configObj->{myHostName};
    $appName = "";  # make sure they didn't pass in something which would hose us.
  }
  elsif ($type eq "App")
  {
    $url = ($appConfigObj->{port} == 80 ? "http" : "https") . "://" . $appConfigObj->{server};
    if (length $appName == 0)
    {
      my @callerInfo = caller;
      ($appName = $callerInfo[0]) =~ s/^(Portal::)([^:]+)(::.+)?$/$2/;
      $appName .= "/";
    }
    else
    {
      if ($appName !~ /\/$/)
      {
        $appName .= "/";  # we need the trailing slash!
      }
    }
  }
  if ($linkType ne "base")
  {
    $url .= $configObj->{siteDir};
    if ($linkType eq "cgi")
    {
      $url .= "/cgi-bin/index.cgi";
    }
    elsif ($linkType eq "cgibin")
    {
      $url .= "/cgi-bin/$appName";
    }
    elsif ($linkType eq "image")
    {
      $url .= "/images/$appName";
    }
    elsif ($linkType eq "js")
    {
      $url .= "/js/$appName";
    }
    elsif ($linkType eq "css")
    {
      $url .= "/css/$appName";
    }
    elsif ($linkType eq "db_dump")
    {
      $url .= "/db_dump/$appName";
    }
    elsif ($linkType eq "install")
    {
      $url .= "/install/$appName";
    }
    elsif ($linkType eq "soap")
    {
      $url .= "/soap/";
    }
    elsif ($linkType eq "soapcgi")
    {
      if ($type eq "App")
      {
        # remove the trailing /
        $appName =~ s/\/$//;
        # now lowercase the appname
        $appName = lc($appName);
      }
      else
      {
        $appName = "portal";
      }
      $url .= "/soap/$appName.cgi";
    }
  }

  return $url;
}

=item DBIWrapper portalDBSetup(type, configObj)

 requires: type, configObj
 returns: DBIWrapper or undef
 type = 'portal' or 'billing'
 configObj = Portal::Data::Config instance.

=cut
sub portalDBSetup
{
  my $self = shift;
  my %args = ( type => "portal", configObj => undef, @_ );
  my $type = $args{type};
  my $configObj = $args{configObj};
  my $db = undef;

  if ($type !~ /^(portal|billing)$/)
  {
    $self->invalid("type", $type);
  }
  if (!defined $configObj)
  {
    $self->missing("configObj");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $db;
  }

  my $dbHost = "";
  my $dbName = "";

  if ($type eq "portal")
  {
    $dbHost = $configObj->dbHost;
    $dbName = $configObj->dbName;
  }
  else
  {
    $dbHost = $configObj->billingdbHost;
    $dbName = $configObj->billingdbName;
  }

  if ($configObj->dbType eq "Postgres")
  {
    $db = DBIWrapper->new(dbHost => $dbHost, dbName => $dbName, dbUser => $configObj->dbUser, dbPasswd => $configObj->dbPasswd, dbPort => $configObj->dbPort, setDateStyle => 0);
  }
  elsif ($configObj->dbType eq "MySQL")
  {
    $db = DBIWrapper->new(dbType => "mysql", dbHost => $dbHost, dbName => $dbName, dbUser => $configObj->dbUser, dbPasswd => $configObj->dbPasswd, dbPort => $configObj->dbPort);
  }
  else
  {
    $self->error("dbType = '$configObj->{dbType}' is invalid!");
  }

  return $db;
}

=item Session portalSession(configObj, portalDB, cookieObj, appName, sessionId, mode)

 requires: configObj
 optional: cookieObj, appName, sessionId, portalDB, mode
 returns: Portal::Session object
 If cookieObj specified, then appName must be specified.
 If sessionType = Database, then portalDB must be defined.

=cut
sub portalSession
{
  my $self = shift;
  my %args = ( configObj => undef, portalDB => undef, cookieObj => undef, appName => "", sessionId => "", mode => "read", @_ );
  my $configObj = $args{configObj};
  my $portalDB = $args{portalDB};
  my $cookieObj = $args{cookieObj};
  my $appName = $args{appName};
  my $sessionId = $args{sessionId};
  my $mode = $args{mode};
  my $session = undef;

  if (!defined $configObj)
  {
    $self->missing("configObj");
  }
  elsif ($configObj->{sessionType} eq "Database" && !defined $portalDB)
  {
    $self->missing("portalDB", " sessionType = 'Database'");
  }
  if ($mode !~ /^(read|write)$/)
  {
    $self->invalid("mode", $mode, "valid values are: read, write!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $session;
  }

  $session = Portal::Session->new(dbHandle => $portalDB, dbType => $configObj->{dbType},
                                    cookieDomain => $configObj->{cookieDomain}, cookieLife => $configObj->{cookieLife},
                                    sessionType => $configObj->{sessionType}, fileDir => $configObj->{sessionDirectory},
                                    lockDir => $configObj->{sessionLockDir},
                                    lockTimeout => $configObj->{lockTimeout}, sessionId => $sessionId,
                                    cookieObj => $cookieObj, appName => $appName, langObj => $self->{langObj}, mode => $mode);
  return $session;
}

=item SessionHandler portalSessionHandler(configObj, portalDB, cookieName, sessionObj, invalidCookieBehaviour)

 requires: configObj
 optional: cookieName, sessionObj, portalDB, invalidCookieBehaviour
 returns: Portal::SessionHandler object
 If cookieName specified, then sessionObj will not be used.
 If cookieName not specified, then sessionObj must be specified.
 If sessionType = Database, then portalDB must be defined.

=cut
sub portalSessionHandler
{
  my $self = shift;
  my %args = ( configObj => undef, portalDB => undef, cookieName => "", sessionObj => undef,
               invalidCookieBehaviour => "1", @_ );
  my $configObj = $args{configObj};
  my $portalDB = $args{portalDB};
  my $cookieName = $args{cookieName};
  my $sessionObj = $args{sessionObj};
  my $invalidCookieBehaviour = $args{invalidCookieBehaviour};
  my $appSessions = undef;

  if (!defined $configObj)
  {
    $self->missing("configObj");
  }
  elsif ($configObj->{sessionType} eq "Database" && !defined $portalDB)
  {
    $self->missing("portalDB");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $appSessions;
  }

  $appSessions = Portal::SessionHandler->new(dbHandle => $portalDB, dbType => $configObj->{dbType},
                                           cookieDomain => $configObj->{cookieDomain}, sessionObj => $sessionObj, cookieLife => $configObj->{cookieLife},
                                           sessionType => $configObj->{sessionType}, fileDir => $configObj->{sessionDirectory}, lockDir => $configObj->{sessionLockDir},
                                           invalidCookieBehaviour => $invalidCookieBehaviour, cookieName => $cookieName, langObj => $self->{langObj},
                                           configObj => $configObj, methods => $self);
  return $appSessions;
}

=item scalar dbCreate(type, user, passwd, host, port, dbName)
 requires: type - MySQL (mysql), Postgres (Pg)
           user - user name to connect with
           passwd - password to use
           host - host to connect to
           port - port to connect to
           dbName - name of database to create
 returns: any output returned from the expect script

=cut
sub dbCreate
{
  my $self = shift;
  my %args = ( type => "", user => "", passwd => "", host => "", port => "", dbName => "", @_ );
  my $type = $args{type};
  my $user = $args{user};
  my $passwd = $args{passwd};
  my $host = $args{host};
  my $port = $args{port};
  my $dbName = $args{dbName};
  my $result = "";

  if ($type !~ /^(Pg|mysql)$/)
  {
    $self->invalid("type", $type);
  }
  if ($user !~ /^(.+)$/)
  {
    $self->invalid("user", $user);
  }
  if ($passwd !~ /^(.+)$/)
  {
    $self->invalid("passwd", $passwd);
  }
  if ($host !~ /^(.+)$/)
  {
    $self->invalid("host", $host);
  }
  if ($port !~ /^(\d+)$/)
  {
    $self->invalid("port", $port);
  }
  if ($dbName !~ /^(.+)$/)
  {
    $self->invalid("dbName", $dbName);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $result;
  }

  my $script = ($type eq "Pg" ? "pg_db_create.exp" : ($type eq "mysql" ? "mysql_db_create.exp" : "pg_db_create.exp" ));
  my $command = "/usr/lib/pcx_portal/$script -host $host -port $port -user $user -pass '$passwd' -dbname $dbName 2>&1";
  $result = `$command`;
  if ($result =~ /(usage:|Error|ERROR)/)
  {
    $self->error("Error Creating database!  Error = '$result'.<br>\n");
    return $result;
  }
  if ($result !~ /Done/)  # check to make sure that we got the string Done back.
  {
    $self->error("Error Creating database!  Done was not returned.  Error = '$result'.<br>\n");
    return $result;
  }

  return $result;
}

=item scalar dbDrop(type, user, passwd, host, port, dbName)
 requires: type - MySQL (mysql), Postgres (Pg)
           user - user name to connect with
           passwd - password to use
           host - host to connect to
           port - port to connect to
           dbName - name of database to create
 returns: undef on error, nothing on success.

=cut
sub dbDrop
{
  my $self = shift;
  my %args = ( type => "", user => "", passwd => "", host => "", port => "", dbName => "", @_ );
  my $type = $args{type};
  my $user = $args{user};
  my $passwd = $args{passwd};
  my $host = $args{host};
  my $port = $args{port};
  my $dbName = $args{dbName};

  if ($type !~ /^(Pg|mysql)$/)
  {
    $self->invalid("type", $type);
  }
  if ($user !~ /^(.+)$/)
  {
    $self->invalid("user", $user);
  }
  if ($passwd !~ /^(.+)$/)
  {
    $self->invalid("passwd", $passwd);
  }
  if ($host !~ /^(.+)$/)
  {
    $self->invalid("host", $host);
  }
  if ($port !~ /^(\d+)$/)
  {
    $self->invalid("port", $port);
  }
  if ($dbName !~ /^(.+)$/)
  {
    $self->invalid("dbName", $dbName);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my $template = ($type eq "Pg" ? "template1" : ($type eq "mysql" ? "mysql" : "unknown" ));
  my $dbObj = DBIWrapper->new(dbType => $type, dbPort => $port, dbHost => $host, dbName => $template, dbUser => $user, dbPasswd => $passwd, autoCommit => 1);
  if ($dbObj->error)
  {
    $self->error($dbObj->errorMessage);
    return undef;
  }
  $dbObj->write(sql => "DROP DATABASE \"$dbName\"");
  if ($dbObj->error)
  {
    $self->error($dbObj->errorMessage);
    return undef;
  }

  return;
}

=item scalar dbDump(type, user, passwd, host, port, dbName, dir, fname)

 requires: type - MySQL (mysql), Postgres (Pg)
           user - user name to connect with
           passwd - password to use
           host - host to connect to
           port - port to connect to
           dbName - name of database to dump
           dir - directory to dump into (make sure it exists)
           fname - file name to dump to
           mode - permissions to create the dump directory with.
                  defaults to 0700
           inserts - 1/0 value. use proper inserts if 1
 returns: any output returned from the expect script

=cut
sub dbDump
{
  my $self = shift;
  my %args = ( type => "", user => "", passwd => "", host => "", port => "", dbName => "", dir => "", fname => "", mode => "0700", inserts => 1, @_ );
  my $type = $args{type};
  my $user = $args{user};
  my $passwd = $args{passwd};
  my $host = $args{host};
  my $port = $args{port};
  my $dbName = $args{dbName};
  my $dir = $args{dir};
  my $fname = $args{fname};
  my $mode = $args{mode};
  my $inserts = $args{inserts};
  my $result = "";

  if ($type !~ /^(Pg|mysql)$/)
  {
    $self->invalid("type", $type);
  }
  if ($user !~ /^(.+)$/)
  {
    $self->invalid("user", $user);
  }
  if ($passwd !~ /^(.+)$/)
  {
    $self->invalid("passwd", $passwd);
  }
  if ($host !~ /^(.+)$/)
  {
    $self->invalid("host", $host);
  }
  if ($port !~ /^(\d+)$/)
  {
    $self->invalid("port", $port);
  }
  if ($dbName !~ /^(.+)$/)
  {
    $self->invalid("dbName", $dbName);
  }
  if ($dir !~ /^(.+\/)$/)
  {
    $self->invalid("dir", $dir, " It must end in a /, but not be '/' only.");
  }
  if ($fname !~ /^(.+)$/)
  {
    $self->invalid("fname", $fname);
  }
  if ($mode !~ /^([0-7]{4})$/)
  {
    $self->invalid("mode", $mode, " It needs to be 4 digits 0 to 7.  See chmod(1) for details.");
  }
  if ($inserts !~ /^(1|0)$/)
  {
    $self->invalid("inserts", $inserts);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $result;
  }

  # buildup the inserts string
  $inserts = ($inserts ? "-inserts" : "");


  # make sure the directory exists.
  if (! -d $dir)
  {
    if (system("/bin/mkdir --mode=$mode -p $dir"))
    {
      $self->error("Error creating dumpDir = '$dir'!  Error = '$!'.<br>\n");
      return $result;
    }
  }

  my $dumpFile = "$dir$fname";
  my $script = ($type eq "Pg" ? "pg_db_dump.exp" : ($type eq "mysql" ? "mysql_db_dump.exp" : "pg_db_dump.exp" ));
  my $command = "/usr/lib/pcx_portal/$script -host $host -port $port -user $user -pass '$passwd' -dbname $dbName -fname $dumpFile $inserts 2>&1";
  $result = `$command`;
  if ($result =~ /(usage:|couldn't open ".+": permission denied|Error|ERROR)/)
  {
    $self->error("Error Dumping database!  Error = '$result'.<br>\n");
    return $result;
  }
  if ($result !~ /Done/)  # check to make sure that we got the string Done back.
  {
    $self->error("Error Dumping database!  Done was not returned.  Error = '$result'.<br>\n");
    return $result;
  }

  # now check and make sure the dumped file was successful.

  # now verify that the dumped file does exist and doesn't contain an error
  if (! -e $dumpFile)
  {
    $self->error("Error Dumping database!  File '$dumpFile' does not exist!<br>\n");
    return $result;
  }

  if ($type eq "Pg")
  {
    $result = `/bin/grep FATAL $dumpFile`;
    if (length $result > 0)
    {
      $self->error("Error Dumping database!  Error = '$result'.<br>\n");
      return $result;
    }
    $result = `/bin/grep ERROR $dumpFile`;
    if (length $result > 0)
    {
      $self->error("Error Dumping database!  Error = '$result'.<br>\n");
      return $result;
    }
    $result = `/bin/grep "Connection to database '$dbName' failed." $dumpFile`;
    if (length $result > 0)
    {
      $self->error("Error Dumping database!  Error = '$result'.<br>\n");
      return $result;
    }
  }
  elsif ($type eq "mysql")
  {
    # I need to add checks for mysql_dump errors.
  }

  return $result;
}

=item scalar dbImport(type, user, passwd, host, port, dbName, fname)

 requires: type - MySQL (mysql), Postgres (Pg)
           user - user name to connect with
           passwd - password to use
           host - host to connect to
           port - port to connect to
           dbName - name of database to work with
           fname - file name to execute SQL commands from
 returns: any output returned from the expect script

=cut
sub dbImport
{
  my $self = shift;
  my %args = ( type => "", user => "", passwd => "", host => "", port => "", dbName => "", fname => "", @_ );
  my $type = $args{type};
  my $user = $args{user};
  my $passwd = $args{passwd};
  my $host = $args{host};
  my $port = $args{port};
  my $dbName = $args{dbName};
  my $fname = $args{fname};
  my $result = "";

  if ($type !~ /^(Pg|mysql)$/)
  {
    $self->invalid("type", $type);
  }
  if ($user !~ /^(.+)$/)
  {
    $self->invalid("user", $user);
  }
  if ($passwd !~ /^(.+)$/)
  {
    $self->invalid("passwd", $passwd);
  }
  if ($host !~ /^(.+)$/)
  {
    $self->invalid("host", $host);
  }
  if ($port !~ /^(\d+)$/)
  {
    $self->invalid("port", $port);
  }
  if ($dbName !~ /^(.+)$/)
  {
    $self->invalid("dbName", $dbName);
  }
  if ($fname !~ /^(.+)$/)
  {
    $self->invalid("fname", $fname);
  }
  # make sure the file exists.
  elsif (! -e $fname)
  {
    $self->invalid("fname", $fname, " File does not exist!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $result;
  }

  my $script = ($type eq "Pg" ? "pg_db_import.exp" : ($type eq "mysql" ? "mysql_db_import.exp" : "pg_db_import.exp" ));
  my $command = "/usr/lib/pcx_portal/$script -host $host -port $port -user $user -pass '$passwd' -dbname $dbName -fname $fname 2>&1";
  $result = `$command`;
  if ($result =~ /(usage:|couldn't open ".+": permission denied|Error|ERROR)/)
  {
    $self->error("Error Importing database!  Error = '$result'.<br>\n");
    return $result;
  }
  if ($result !~ /Done/)  # check to make sure that we got the string Done back.
  {
    $self->error("Error Importing database!  Done was not returned.  Error = '$result'.<br>\n");
    return $result;
  }

  return $result;
}

=item scalar urlBuilder(doc, baseUrl, arguments)

 requires: baseUrl - the base url to append the arguments to.
           arguments - hash of name => value pairs
 optional: doc - if not specified we instantiate an instance
           of HTMLObject::Base to use

 arguments format:
   if the value is a scalar then the output is name=value,
   else if it is an array then the output is name=value[0]&
   name=value[1], etc.

 The result is a string of name=value where the value is
 uri encoded and each pair is & seperated.

 Returns a string consisting of the url ? arguments.
 Ex: baseUrl = http://www.test.com/portal/cgi-bin/index.cgi
     arguments = { app => "Accounting", state => "Main",
                   command => "display" }
 returns:
 http://www.test.com/portal/cgi-bin/index.cgi?app=Accounting&state=Main&command=display

=cut
sub urlBuilder
{
  my $self = shift;
  my %args = ( doc => undef, baseUrl => "", arguments => undef, @_ );
  my $doc = $args{doc};
  my $baseUrl = $args{baseUrl};
  my $arguments = $args{arguments};
  my $result = "";

  if (!defined $doc)
  {
    $doc = HTMLObject::Base->new();
  }
  if (!defined $arguments)
  {
    $self->missing("arguments");
  }
  if (length $baseUrl == 0)
  {
    $self->missing("baseUrl");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $result;
  }

  my %specialCases = ();  # used to keep track of app, state, arg being processed.
  foreach my $name ("app", "state", "command")
  {
    if (exists $arguments->{$name})
    {
      $specialCases{$name} = 1;
      $result .= "&" if ($result);  # make sure we have the seperating &.

      my $value = $arguments->{$name};
      my $uriName = $doc->encodeString(string => $name);
      if (ref($value) eq "ARRAY")
      {
        my $count = 0;
        foreach my $subValue (@{$value})
        {
          $result .= "&" if ($count);  # make sure we have the seperating & between array values.
          $result .= "$uriName=" . $doc->encodeString(string => $subValue);
          $count++;
        }
      }
      else
      {
        $result .= "$uriName=" . $doc->encodeString(string => $value);
      }
    }
  }

  foreach my $name (keys %{$arguments})
  {
    if (! exists $specialCases{$name})
    {
      $result .= "&" if ($result);  # make sure we have the seperating &.

      my $value = $arguments->{$name};
      my $uriName = $doc->encodeString(string => $name);
      if (ref($value) eq "ARRAY")
      {
        my $count = 0;
        foreach my $subValue (@{$value})
        {
          $result .= "&" if ($count);  # make sure we have the seperating & between array values.
          $result .= "$uriName=" . $doc->encodeString(string => $subValue);
          $count++;
        }
      }
      else
      {
        $result .= "$uriName=" . $doc->encodeString(string => $value);
      }
    }
  }

  return "$baseUrl?$result";
}

sub formatDateOutput
{
  my $self = shift;
  my %args = ( @_ );
  my $tm = $args{tm};
  my $format = $args{format};
  my $result = "";

  $result = strftime($format, @{$tm});

  return $result;
}

=item scalar getCurrentDate(format)

 Uses localtime to gather the current date and returns a string
 with the specified parameters output in the format specified
 by format.

 By default format = "%F" which returns YYYY-MM-DD.

 See strftime(3) for possible arguments.

=cut

sub getCurrentDate
{
  my $self = shift;
  my %args = ( format => "%F", @_ );

  my @tm = localtime;
  my $result = $self->formatDateOutput(tm => \@tm, %args);

  return $result;
}

=item scalar getCurrentLocalizedDate(tz, format)

 Uses gmtime to gather the current date and returns a string
 with the specified parameters output in the format specified
 by format.

 By specifying your TimeZone in tz, I can calculate your
 localized time.  tz will default to "GMT".

 By default format = "%F" which returns YYYY-MM-DD.

 See strftime(3) for possible arguments.

=cut

sub getCurrentLocalizedDate
{
  my $self = shift;
  my %args = ( format => "%F", tz => "GMT", @_ );

  my @tm = gmtime;

  if ($args{tz} ne "GMT")
  {
    @tm = $self->calculateDateOffset(date => scalar(gmtime), offset => "0 hours", toTZ => $args{tz}, fromTZ => "GMT");
  }

  my $result = $self->formatDateOutput(tm => \@tm, %args);

  return $result;
}

=item @tm calculateDateOffset(date, offset, toTZ, fromTZ)

  Takes the date as given by the user (can be "now" to indicate
  they want the current time used) and uses Date::Manip DateCalc
  method to calculate the new date.

  toTZ indicates the TimeZone to convert into.
  fromTZ indicates the TimeZone we are converting from.  If "",
    then it is the servers default timezone.  It defaults to "".

  toTZ defaults to "".  This will calculate the date in the servers
    current TimeZone.  If you specify a TimeZone, then we convert
    to that TimeZone from the fromTZ value.

  Returns the updated tm array representing the new date/time just
  calculated.

=cut

sub calculateDateOffset
{
  my $self = shift;
  my %args = ( date => "now", offset => "0 hours", toTZ => "", fromTZ => "", @_ );
  my $err;
  my @tm = ();

  my $time = DateCalc($args{date}, $args{offset}, \$err);
  if ($time)
  {
    # see if we have to convert to a specific TimeZone.
    if ($args{toTZ} ne "")
    {
      # get the number of seconds past the epoch.
      $time = UnixDate($time, "%s");

      # convert back to the tm array.
      @tm = localtime($time);
      $time = $self->formatDateOutput(tm => \@tm, format => "%Y-%m-%d %H:%M:%S");  # force the format so that we always have a valid date/time back.
      $time = ParseDate($time);
      # I have to jump through hoops to have the input be a ParseDate object. :(
      $time = Date_ConvTZ($time, $args{fromTZ}, $args{toTZ});
    }

    # get the number of seconds past the epoch.
    $time = UnixDate($time, "%s");

    # convert back to the tm array.
    @tm = localtime($time);
  }

  return @tm;
}

=item scalar formatDateString(date, format)

 Runs the specified date through Date::Manip ParseDate
 and then returns it as specified by the format string.

 format defaults to "%F".

 date defaults to "now".

=cut

sub formatDateString
{
  my $self = shift;
  my %args = ( date => "now", format => "%F", @_ );
  my $result = "";

  my $date = ParseDate($args{date});
  if ($date)
  {
    my $time = UnixDate($date, "%s");

    my @tm = localtime($time);

    $result = $self->formatDateOutput(tm => \@tm, %args);
  }

  return $result;
}

=item scalar getDateOffset(date, offset, format)

 Returns the date/time specified in the format string that is
 the current date/time or specified date offset by the
 specified offset string that is valid for
 Date::Manip::ParseDate() function.

 format defaults to "%F" - YYYY-MM-DD

 date defaults to "now" which will use the current time for
 any calculations.

=cut

sub getDateOffset
{
  my $self = shift;
  my %args = ( format => "%F", date => "now", offset => "", @_ );

  my @tm = $self->calculateDateOffset(%args);

  my $result = $self->formatDateOutput(tm => \@tm, %args);

  return $result;
}

=item bool isValidDate(date, seperator)

 Returns 1 if the date is a valid ISO format YYYY-MM-DD,
 else 0.

 You can specify the seperator to check.  Defaults to -.

=cut

sub isValidDate
{
  my $self = shift;
  my %args = ( date => "", seperator => "-", @_ );
  my $date = $args{date};
  my $seperator = $args{seperator};

  $seperator = "\\" . $seperator;

  if ($date !~ /^(\d{4})$seperator(\d{2})$seperator(\d{2})$/)
  {
    return 0;
  }
  return 1;
}

=item (bool, date, error) fixupDate(date, seperator, year)

 Returns an array where index 0 is the boolean result
 of 1 = valid, 0 = invalid and index 1 is the date.
 Index 2 will contain the error message if invalid.

 The date, if valid, will be returned in ISO format
 of YYYY-MM-DD using the specified seperator instead of -.

 If only 2 parts are specified and they are both 1 or 2
 digits, then we use the specified year to make it a valid
 date, but only if year > 0 and is 4 digits.

=cut

sub fixupDate
{
  my $self = shift;
  my %args = ( date => "", seperator => "-", year => 0, @_ );
  my $date = $args{date};
  my $seperator = $args{seperator};
  my $year = $args{year};
  my $tmpDate = $date;
  my @result = (0, $date, "");

  # make sure the seperator is one of the allowed values.
  if ($seperator !~ /^[-\\\/]$/)
  {
    $result[2] = "fixupDate: seperator='$seperator' is invalid!";
    return @result;
  }
  if ($year !~ /^(0|\d{4})$/)
  {
    $result[2] = "fixupDate: year='$year' is invalid!";
    return @result;
  }
  if (!$self->isValidDate(date => $date, seperator => $seperator))
  {
    # replace any non digit/seperator characters with nothing.
    $tmpDate =~ s/[^0-9\-\\\/]//g;

    if (length $tmpDate == 0)
    {
      $result[2] = "fixupDate: Date='$date' is invalid!";
      $result[1] = $tmpDate;
      return @result;
    }

    # fixup all seperators to be -.
    $tmpDate =~ s/[-\\\/]/-/g;

    my @digits = split("-", $tmpDate);
    # figure out what we have to work with.
    if (@digits == 0)
    {
      $result[2] = "fixupDate: Date='$tmpDate' has no $seperator seperators!";
      $result[1] = $tmpDate;
      return @result;
    }
    elsif (@digits > 3)
    {
      $result[2] = "fixupDate: Date='$tmpDate' has more than 2 $seperator seperators!";
      $result[1] = $tmpDate;
      return @result;
    }
    elsif (@digits == 1)
    {
      $result[2] = "fixupDate: Date='$tmpDate' does not have enough data to work with!";
      $result[1] = $tmpDate;
      return @result;
    }
    elsif (@digits == 2)
    {
      # make sure we have 2, 2 or 1 digit values which we can then fixup and tack the year on.
      if (length $digits[0] > 2)
      {
        $result[2] = "fixupDate: Date='$tmpDate' does not have a month part!";
        $result[1] = $tmpDate;
        return @result;
      }
      if (length $digits[1] > 2)
      {
        $result[2] = "fixupDate: Date='$tmpDate' does not have a day part!";
        $result[1] = $tmpDate;
        return @result;
      }
      if ($year == 0)
      {
        $result[2] = "fixupDate: Date='$tmpDate' does not have a year part and no default year specified!";
        $result[1] = $tmpDate;
        return @result;
      }
      $tmpDate = $year . $seperator . (length $digits[0] == 1 ? "0" : "") . $digits[0] . $seperator . (length $digits[1] == 1 ? "0" : "") . $digits[1];
    }
    elsif (@digits == 3)
    {
      # make sure it is year, month, day
      if (length $digits[0] <= 2 && length $digits[1] <= 2 && length $digits[2] == 4)
      {
        $tmpDate = $digits[2] . $seperator . (length $digits[0] == 1 ? "0" : "") . $digits[0] . $seperator . (length $digits[1] == 1 ? "0" : "") . $digits[1];
      }
      else
      {
        if (length $digits[0] == 4)
        {
          if (length $digits[1] > 2 || length $digits[2] > 2)
          {
            $result[2] = "fixupDate: Date='$tmpDate' is invalid!";
            $result[1] = $tmpDate;
            return @result;
          }
          else
          {
            $tmpDate = $digits[0] . $seperator . (length $digits[1] == 1 ? "0" : "") . $digits[1] . $seperator . (length $digits[2] == 1 ? "0" : "") . $digits[2];
          }
        }
      }
    }
  }

  $result[0] = 1;
  $result[1] = $tmpDate;
  return @result;
}

=item bool doLog(db, action, extraInfo, sessionObj, userId, logObj)

 requires: db, action
 optional: extraInfo, sessionObj, userId, logObj
 returns: 1 = OK, 0 = error
 summary: This routine generates the LogEntry, LogObj and then
          inserts it.

 Or you can define the userId and not specify sessionObj.

 If you specify your own logObj, then it will be used
 instead of instantiating one, thus db does not need to
 be specified in this case.

=cut
sub doLog
{
  my $self = shift;
  my %args = ( db => undef, action => -1, extraInfo => "", sessionObj => undef, userId => "", logObj => undef, @_ );
  my $db = $args{db};
  my $action = $args{action};
  my $extraInfo = $args{extraInfo};
  my $sessionObj = $args{sessionObj};
  my $userId = $args{userId};
  my $logObj = $args{logObj};

  if ($action !~ /^(\d+)$/)
  {
    $self->invalid("action", $action);
  }
  if (!defined $db && !defined $logObj)
  {
    $self->missing("db");
  }
  elsif (defined $db && !$db->isValid)
  {
    $self->error($db->errorMessage);
  }
  if (!defined $sessionObj && $userId !~ /^(-1|\d+)$/)
  {
    $self->missing("sessionObj");
  }
  elsif (defined $sessionObj && !$sessionObj->isValid)
  {
    $self->error($sessionObj->errorMessage);
  }
  if (defined $logObj && !$logObj->isValid)
  {
    $self->error($logObj->errorMessage);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  if ($self->error)
  { # catch the plain error cases.
    return 0;
  }

  my $logEntry = Portal::Objects::LogEntry->new(action => $action, ipAddress => (exists $ENV{REMOTE_ADDR} ? $ENV{REMOTE_ADDR} : '127.0.0.1'), userId => (defined $sessionObj ? $sessionObj->{store}->{userObj}->{id} : (length $userId ? $userId : -1)), extraInfo => $extraInfo, langObj => $self->{langObj});
  if ($logEntry->error)
  {
    $self->error($logEntry->errorMessage);
    return 0;
  }

  # now instantiate the Log object.
  if (!defined $logObj)
  {
    $logObj = Portal::Log->new(dbHandle => $db, langObj => $self->{langObj});
    if ($logObj->error)
    {
      $self->error($logObj->errorMessage);
      return 0;
    }
  }

  # now log it
  $logObj->newEntry(logEntry => $logEntry);
  if ($logObj->error)
  {
    $self->error($logObj->errorMessage);
    return 0;
  }

  return 1;
}

=item scalar getConfigValue(name, portalDB)

 requires: name, portalDB
 optional:
 returns:  the value associated w/ name in the config_tb table or
           undef if the name does not exist.

=cut
sub getConfigValue
{
  my $self = shift;
  my %args = (name => "", portalDB => undef, @_);
  my $name = $args{name};
  my $portalDB = $args{portalDB};
  my $value = undef;

  if (length $name == 0)
  {
    $self->invalid("name", $name);
  }
  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return $value;
  }

  my $sql = "SELECT value FROM config_tb WHERE name = ?";
  my $sth = $portalDB->read(sql => $sql, plug => [ $name ]);
  if ($portalDB->error)
  {
    $self->error($portalDB->errorMessage);
    return;
  }
  my @info = $sth->fetchrow_array;
  if (defined $info[0])
  {
    $value = $info[0];
  }

  return $value;
}

=item scalar getConfigDefault(app, module, prefName, portalDB)

 looks up the config entry for the specified app, module
 and prefName.  Handles the case where it doesn't exist, etc.

 returns the value in the config system or undef on error.

 If the entry doesn't exist, then we do not set an error and
 return undef.

=cut
sub getConfigDefault
{
  my $self = shift;
  my %args = (app => "", module => "", prefName => "", portalDB => undef, @_);
  my $callingApp = $args{app};
  my $prefModule = $args{module};
  my $prefName = $args{prefName};
  my $portalDB = $args{portalDB};

  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  if (length $callingApp == 0)
  {
    $self->missing("callingApp");
  }
  if (length $prefModule == 0)
  {
    $self->missing("prefModule");
  }
  if (length $prefName == 0)
  {
    $self->missing("prefName");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  # get the default for this preference
  my $prefName = $callingApp . "_" . $prefModule . "_" . $prefName;
  my $prefValue = $self->getConfigValue(name => $prefName, portalDB => $portalDB);
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  return $prefValue;
}

=item int setConfigValue(name, value, log, sessionObj, portalDB)

 requires: name, sessionObj, portalDB
 optional: value, log = 1
 returns: 1 - value set, 0 - error, -1 - name already exists,
          -2 - permission denied.

=cut
sub setConfigValue
{
  my $self = shift;
  my %args = (name => "", value => "", log => 1, sessionObj => undef, portalDB => undef, @_);
  my $name = $args{name};
  my $value = $args{value};
  my $log = $args{log};
  my $sessionObj = $args{sessionObj};
  my $portalDB = $args{portalDB};

  if (length $name == 0)
  {
    $self->invalid("name", $name);
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->invalid("log", $log);
  }
  if (!defined $sessionObj)
  {
    $self->missing("sessionObj");
  }
  elsif (!$sessionObj->isValid)
  {
    $self->invalid("sessionObj", "not valid");
  }
  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  elsif (!$portalDB->isValid)
  {
    $self->invalid("portalDB", "not valid");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  my $userObj = $sessionObj->{store}->{userObj};

  # make sure we are a sysadmin
  if ($userObj->{sysadmin} == 0)
  {
    return -2;
  }

  # first make sure this entry doesn't exist.
  my $tempEntry = $self->getConfigValue(name => $name, portalDB => $portalDB);
  if (not defined $tempEntry)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # config entry doesn't exist yet!
    {
      $portalDB->write(sql => "INSERT INTO config_tb (name, value) VALUES (?, ?)",
                               plug => [ $name, $value ]);
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }
      $portalDB->commit;  # make the changes permanent.
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Config Entry Create event.
        $self->doLog(sessionObj => $sessionObj, db => $portalDB, action => 27, extraInfo => "Config Entry: name='$name', value='$value'");
        if ($self->error)
        {
          $self->prefixError();
          return 0;
        }
      }
    }
  }
  else
  {
    return -1;
  }

  return 1;
}

=item int updateConfigValue(name, value, log, sessionObj, portalDB)

 requires: name, sessionObj, portalDB
 optional: value, log = 1
 returns: 1 - value set, 0 - error, -1 - name does not exist,
          -2 - permission denied

=cut
sub updateConfigValue
{
  my $self = shift;
  my %args = (name => "", value => "", log => 1, sessionObj => undef, portalDB => undef, @_);
  my $name = $args{name};
  my $value = $args{value};
  my $log = $args{log};
  my $sessionObj = $args{sessionObj};
  my $portalDB = $args{portalDB};

  if (length $name == 0)
  {
    $self->invalid("name", $name);
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->invalid("log", $log);
  }
  if (!defined $sessionObj)
  {
    $self->missing("sessionObj");
  }
  elsif (!$sessionObj->isValid)
  {
    $self->invalid("sessionObj", "not valid");
  }
  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  elsif (!$portalDB->isValid)
  {
    $self->invalid("portalDB", "not valid");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  my $userObj = $sessionObj->{store}->{userObj};

  # make sure we are a sysadmin
  if ($userObj->{sysadmin} == 0)
  {
    return -2;
  }

  # first make sure this entry does exist.
  my $tempEntry = $self->getConfigValue(name => $name, portalDB => $portalDB);
  if (defined $tempEntry)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # config entry does exist!
    {
      $portalDB->write(sql => "UPDATE config_tb SET value = ? WHERE name = ?",
                               plug => [ $value, $name ]);
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }
      $portalDB->commit;  # make the changes permanent.
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Config Entry Create event.
        $self->doLog(sessionObj => $sessionObj, db => $portalDB, action => 28, extraInfo => "Config Entry: name='$name', value='$value'");
        if ($self->error)
        {
          $self->prefixError();
          return 0;
        }
      }
    }
  }
  else
  {
    return -1;
  }

  return 1;
}

=item int deleteConfigValue(name, log, sessionObj, portalDB)

 requires: name, sessionObj, portalDB
 optional: log = 1
 returns: 1 - item deleted, 0 - error, -1 - name does not exist,
          -2 - permission denied

=cut
sub deleteConfigValue
{
  my $self = shift;
  my %args = (name => "", log => 1, sessionObj => undef, portalDB => undef, @_);
  my $name = $args{name};
  my $log = $args{log};
  my $sessionObj = $args{sessionObj};
  my $portalDB = $args{portalDB};

  if (length $name == 0)
  {
    $self->invalid("name", $name);
  }
  if ($log !~ /^(1|0)$/)
  {
    $self->invalid("log", $log);
  }
  if (!defined $sessionObj)
  {
    $self->missing("sessionObj");
  }
  elsif (!$sessionObj->isValid)
  {
    $self->invalid("sessionObj", "not valid", $sessionObj->errorMessage);
  }
  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  elsif (!$portalDB->isValid)
  {
    $self->invalid("portalDB", "not valid", $portalDB->errorMessage);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }
  my $userObj = $sessionObj->{store}->{userObj};

  # make sure we are a sysadmin
  if ($userObj->{sysadmin} == 0)
  {
    return -2;
  }

  # first make sure this entry does exist.
  my $tempEntry = $self->getConfigValue(name => $name, portalDB => $portalDB);
  if (defined $tempEntry)
  {
    if ($self->error)
    {
      $self->prefixError();
      return 0;
    }
    else  # config entry does exist!
    {
      $portalDB->write(sql => "DELETE FROM config_tb WHERE name = ?",
                               plug => [ $name ]);
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }
      $portalDB->commit;  # make the changes permanent.
      if ($portalDB->error)
      {
        $self->error($portalDB->errorMessage);
        return 0;
      }

      if ($log)
      { # log the Config Entry Create event.
        $self->doLog(sessionObj => $sessionObj, db => $portalDB, action => 29, extraInfo => "Config Entry: name='$name', value='$tempEntry'");
        if ($self->error)
        {
          $self->prefixError();
          return 0;
        }
      }
    }
  }
  else
  {
    return -1;
  }

  return 1;
}

=item % getAllConfigItems(portalDB)

 requires: portalDB
 optional:
 returns: hash of all config_tb entries as (name, value) pairs.

=cut
sub getAllConfigItems
{
  my $self = shift;
  my %args = ( portalDB => undef, @_ );
  my $portalDB = $args{portalDB};
  my %items = ();

  if (!defined $portalDB)
  {
    $self->missing("portalDB");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  my $sql = "SELECT name, value FROM config_tb ORDER BY name";
  my $sth = $portalDB->read(sql => $sql);
  if ($portalDB->error)
  {
    $self->error($portalDB->errorMessage);
    return %items;
  }
  while (my @info = $sth->fetchrow_array)
  {
    $items{$info[0]} = $info[1];
  }

  return %items;
}

=item void setJavaScriptErrorInfo(doc, email, appName, appVersion)

 requires: doc, email, appName, appVersion
 returns: nothing

 Updates the specified doc (HTMLObject::Normal document)'s
 javascript error handler variables.

=cut
sub setJavaScriptErrorInfo
{
  my $self = shift;
  my %args = ( doc => undef, email => "", appName => "", appVersion => "", @_ );
  my $doc = $args{doc};
  my $email = $args{email};
  my $appName = $args{appName};
  my $appVersion = $args{appVersion};

  if (!defined $doc)
  {
    $self->missing("doc");
  }
  if ($email !~ /^([^@]+\@[^@\.]+\.[^@]+)$/)
  {
    $self->invalid("email", $email);
  }
  if ($appName !~ /^(.+)$/)
  {
    $self->invalid("appName", $appName);
  }
  if ($appVersion !~ /^(.+)$/)
  {
    $self->invalid("appVersion", $appVersion);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return;
  }

  $doc->setJavascriptErrorHandlerEmail(email => $email);
  $doc->setJavascriptErrorHandlerPrgName(name => "$appName");
  $doc->setJavascriptErrorHandlerPrgVersion(version => $appVersion);
}

=item @MenuItem createMenuEntry(description, portalSession, sessionObj, appName, helpType, helpTopics, refreshArgs)

 requires: description, portalSession, sessionObj, appName
 optional: helpType, helpTopics, refreshArgs
 returns:  array of Portal::Objects::MenuItem's

 description is a string of menu entries you want created,
   seperated by commas.  Possible entries are:
   close, help, refresh, seperator, user, print

   close - generates the Close App menu entry
   help  - generates the Help menu entry
   refresh - generates the Refresh menu entry
   seperator - generates a seperator menu entry
   user - outputs the current user info (3 menu entries)
   print - generate a Print document link

 helpType is either Normal or System.  Defaults to 'Normal'.

 helpTopics is the string of topics that the Help engine will
   drill down to when you click on Help.

 refreshArgs is the hash of arguments that should be added to
   the refresh url to make sure the correct app/state/command
   is re-executed.

 Ex: @leftItems = $self->{methods}->createMenuEntry(
       description => "close, seperator, help, seperator, refresh",
       portalSession => $self->{portalSession},
       sessionObj => $self->{sessionObj}, appName => $self->{appName},
       refreshArgs => { windowName => "xyz" });

   would generate an array of MenuItems where you had the first
   being the Close App entry, then a seperator, then the Help
   entry (no special args), then a seperator, then the Refresh
   entry ( with extra argument windowName=xyz added).

=cut
sub createMenuEntry
{
}

=item soapObj getPortalSOAPObj(caller, configObj, location, companyAppObj)

 location = 'Portal' or 'appName'.  If 'Portal', then we are connecting
   to just the Portal::SOAP module.
   If not 'Portal', then we are dealing with a the Portal::<APP>::SOAP
   module where <APP> is the value of location, and the companyAppObj
   is required.  Ex:  location='Accounting' would deal with XIWA's
   Portal::Accounting::SOAP module.

 Creates an instance of the Portal::SOAP module or
   Portal::<appName>::SOAP module and returns it to you to work with
   after calling new() on it.  It is upto you to verify the object is
   valid.

 Returns undef on error.

 This code is not working 100% and is only here as a starting point
 for future development.  The SOAP::Lite infrastructure is not
 being used in this version.

=cut
sub getPortalSOAPObj
{
  my $self = shift;
  my %args = ( caller => undef, configObj => undef, location => "", companyAppObj => undef, @_ );
  my $caller = $args{caller};
  my $configObj = $args{configObj};
  my $location = $args{location};
  my $companyAppObj = $args{companyAppObj};

  if (!defined $caller)
  {
    $self->missing("caller");
  }
  elsif (!$caller->isValid)
  {
    $self->invalid("caller", $caller->errorMessage);
  }
  if (!defined $configObj)
  {
    $self->missing("configObj");
  }
  elsif (!$configObj->isValid)
  {
    $self->invalid("configObj", $configObj->errorMessage);
  }
  if ($location ne "Portal")
  {
    if (!defined $companyAppObj)
    {
      $self->missing("companyAppObj");
    }
    elsif (!$companyAppObj->isValid)
    {
      $self->invalid("companyAppObj", $companyAppObj->errorMessage);
    }
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return undef;
  }

  my $uri = "";
  my $proxy = "";
  my $module = "";
  if ($location eq "Portal")
  {
    $uri = $self->createBaseURL(type => 'Portal', linkType => 'base');
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
    $uri .= "Portal/SOAP";

    $proxy = $self->createBaseURL(type => 'Portal', linkType => 'soapcgi');
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }

    $module = "Portal::SOAP";
  }
  else
  {
    # build a url for working with the app in question.
    $uri = $self->createBaseURL(type => 'Portal', linkType => 'base');
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
    $uri .= "Portal/$location/SOAP";

    $proxy = $self->createBaseURL(type => 'App', linkType => 'soapcgi',
      appConfigObj => $companyAppObj, appName => $location);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }

    $module = "Portal::$location" . "::SOAP";
  }

#  eval "use SOAP::Lite dispatch_from => '$module', uri => '$uri', proxy => ['$proxy', options => {compress_threshold => 10000}], on_fault => sub { my ( \$soap, \$res ) = \@_; die ref \$res ? \$res->faultstring : \$soap->transport->status, \"\n\"; }; ";
#  if ($@)
#  {
#    $self->error($@);
#    return undef;
#  }

  eval "use $module;";
  if ($@)
  {
    $self->error("use $module failed: $@");
    return undef;
  }

  my $portalSoapObj = undef;
  eval " \$portalSoapObj = $module" . "->new(langObj => \$self->{langObj},
    methods => \$self, configObj => \$configObj, caller => \$caller); ";

  if ($@)
  {
    $self->error($module . "->new failed: $@");
    return undef;
  }

  if (!defined $portalSoapObj)
  {
    $self->error("portalSoapObj is not defined!");
    return undef;
  }

  return $portalSoapObj;
}

=item scalar getHandlerSub(index)

  This method does not handle pass by name semantics.  Just
  specify your string as the argument:
  Ex:  $self->{methods}->getHandlerSub("User");

 requires: index - string to index into handler hash with to
   determine what handler subroutine to return.

 Available handlers are:
   User
   AppServerObject
   'Name = Value'  - sets the name = value for each entry.


 returns: handler sub for the specified index.
   If the index does not exist, we return undef.

=cut
sub getHandlerSub
{
  my $self = shift;
  my $index = shift;

  my %handlers = (
    "AppServerObject" => sub { my $a = shift; my %b = (); $b{names} = $a->{server} . ":" . $a->{port} . " ($a->{dbType} - " . $a->{dbHost} . ":" . $a->{dbPort} . ")"; $b{values} = $a->{counter}; return %b; },
    "User" => sub { my $a = shift; my %b = (); $b{names} = $a->{uname} . " (" . $a->{fname} . " " . $a->{mname} . " " .  $a->{lname} . ") [" . ($a->{active} ? "" : "*") . ($a->{admin} ? "A" : "") . ($a->{sysadmin} ? "S" : "") . "]"; $b{values} = $a->{uname}; return %b; },
    "Name = Value" => sub { my $a = shift; my %b = (); $b{names} = $a; $b{values} = $a; return %b; },
    );

  if (exists $handlers{$index})
  {
    return $handlers{$index};
  }

  return undef;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Objects::MenuItem(3), Portal::Base(3)

=cut

