# Base.pm - The Object Class that provides a Base Object
# Created by James A. Pattie, 2004-12-13.
# Last edited 2004-12-13.

# Copyright (c) 2000-2005 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::Base;
use strict;
use overload '==' => \&objsEqual,
             '!=' => \&objsNotEqual,
             '""' => \&printStr;
use Portal::Base;
use Portal::Data::Variables;
use HTMLObject::Form;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.02';

=head1 NAME

Base - Object used to build a base Object Class.

=head1 SYNOPSIS

 package Portal::Objects::Test;
 use Portal::Objects::Base;
 use strict;
 use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

 require Exporter;

 @ISA = qw(Portal::Objects::Base Exporter AutoLoader);
 @EXPORT = qw();

 $VERSION = '0.01';

 sub new
 {
   my $class = shift;
   my $self = $class->SUPER::new(@_);
   my %args = ( something => 'Hello World!', @_ );

   if ($self->didErrorOccur)
   {
     $self->prefixError();
     return $self;
   }

   # specify whether or not we are using the HTMLObject::Form.
   $self->{usingForms} = 0;

   # specify whether or not we need to still generate the
   # $self->{variablename} entries in populate() to support old
   # code that is still referencing $obj->{variablename} instead
   # of doing $obj->variablename().
   $self->{backwardsCompatible} = 1;

   # instantiate anything unique to this module
   $self->{objName} = "Test";   # specify your objects name here.

   # define your data variables and their types.
   # this must be in the form called main, even if you are not using
   # forms or do not want to display a form on the screen called main,
   # the only way for me to know what entries in the object are data
   # elements and to know what types they are is to define them in
   # the data->{main} hash.  At a minimum, you must have the -DataType and
   # -Value entries defined, though if you want to use main to display
   # via the HTMLObject::Form module, you will need to define the other
   # attributes like -Type, -Label, etc.
   #
   # Do NOT add -DataType to entries that are only existing for the purpose
   # of the form, like app, state, command.
   # Only those entries that have -DataType in them will be updated by the
   # populate() method, thus keeping the data initialization to just
   # the actual data elements of this object.  You can set any element
   # that exists in data->{main} via the AUTOLOAD code, as long as it has
   # the -DataType attribute set, like:
   # $obj->app("MyApp");

   $self->{data}->{main} =
     { "fname" => { -DataType => "string", -Value => "" },
       "lname" => { -DataType => "string", -Value => "" },
       "age" => { -DataType => "int", -Value => ""},
     };

   # do validation - ONLY if not being derived from!
   # The $self->Portal::Objects::Test::isValid makes sure we access our
   # isValid method and not any classes isValid method that has
   # derived from us.
   if (!$self->Portal::Objects::Test::isValid)
   {
     # the error is set in the isValid() method.
     return $self;
   }

   # do anything else you might need to do.
   return $self;
 }

 sub isValid
 {
   my $self = shift;

   # make sure our Parent class is valid.
   if (!$self->SUPER::isValid())
   {
     $self->prefixError();
     return 0;
   }

   # validate our parameters.
   if ($self->{fname} !~ /^(.+)$/)
   {
     $self->invalid("fname", $self->{fname});
   }

   if ($self->numInvalid() > 0 || $self->numMissing() > 0)
   {
     $self->error($self->genErrorString("all"));
     return 0;
   }
   return 1;
 }

=head1 DESCRIPTION

Base is a base object class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(void)

 Creates a new instance of the Portal::Objects::Base module.
 See Portal::Base(3) for a listing of required arguments.

 Variables defined by this class:

 backwardsCompatible - boolean value that indicates if the
   populate method needs to generate the $self-> entries
   for code that has not yet been updated to reference the
   data by $obj->get()/set().  Defaults to 0 (false) to force
   the new behaviour.

 objName - Name of the object ("Base", "User", etc.)
 data - hash of data structures used by the HTMLObject::Form.
   Each entry is named after the form.
   The entry named 'main' is special in that it defines the
   elements that this Object stores and will use for
   object comparison and display.
 data->{main} - hash of variables that make up the actual data
   elements of this object.
   The keys are the names of the variables and the value is an
   anonymous hash with at least the following attributes:
     -DataType - specifies the type of the element for comparison
       purposes.  Valid types are:
         string
         int
         float
         hash
         array
         object
      Currently only the string, int and float types are
      used when checking for object equality and displaying
      the object as text, html or xml.

      DO NOT specify -DataType if the element being defined in
      data->{main} is a form only element and is not considered
      a data element for this object.  Examples are:
        app, state, command

      By specifying -DataType, you are telling me that the entry in
      question is considered data for the object and will be
      used when printing and comparing this object.

     -Value - specifies the value associated with this element.
       If specified when defining the 'main' form then this will
       be the default value for this element, thus eliminating the
       need to define default values for %args in the populate()
       method.

   Ex:
   $self->{data}->{main} =
     { "uname" => { -DataType => "string", -Value => "" },
       "admin" => { -DataType => "int", -Value => "0" },
       "passwd" => { -DataType => "string", -Value => "" },
       "colors" => { -DataType => "array", -Value => [ "blue", "red"] },
       "app" => { -Type => "hidden", -Value => "MyApp" },
       "state" => { -Type => "hidden", -Value => "MyState" },
       "command" => { -Type => "hidden", -Value => "display" },
     };

   Would define the following elements as being data elements for
   this object: uname, admin, passwd, colors.

   app, state, command would be form only elements since they do
   not have -DataType specified in their definitions.

   To have this definition be a valid form you would need to add
   the -Type attributes to uname, admin, passwd, and colors along
   with any other attributes needed to properly display them.

 variables - Portal::Data::Variables object.

 formObj - HTMLObject::Form object.

 usingForms - boolean that indicates if the object is using
   the HTMLObject::Form for manipulation and/or displayall.
   Defaults to 1 (true).

 template - hash of templates used by the HTMLObject::Form.
   Each entry is named after the form.
 profile - hash of profile structures used by the HTMLObject::Form.
   Each entry is named after the form.
 order - hash of arrayrefs that define the disply order for the
   form items in each form, if they are using createTemplate().
   This has to be created, but can be empty arrays for each form.
 javascript - hash of javascript code blocks grouped by form name,
   which are passed into the generate() method.

 Even if your code only is using one form, you still have to name it
 and index the template, data and profile hashes with the form name.
 This allows for easy expansion in the future, without having to
 go back and name forms later, etc.

 method - the default method forms are submitted as.
   Valid values are: post, get
   Default value is 'post'.

 action - the default action string for your forms.
   Default value is ''.


 You now have to use the populate() method to assign values to the
 parameters and have them be validated.

 new() just instantiates the object and does not do any initial
 assigning to forms, variables, etc.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = (  @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{backwardsCompatible} = 0;
  $self->{usingForms} = 1;
  $self->{template} = {};
  $self->{data} = {};
  $self->{data}->{main} = {};  # pre-create the 'main' form hash for the user.
  $self->{profile} = {};
  $self->{order} = {};
  $self->{objName} = "Base";
  $self->{method} = "post";
  $self->{action} = "";

  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  if ($self->{variables}->error)
  {
    $self->error($self->{variables}->errorMessage);

    return $self;
  }

  $self->{formObj} = HTMLObject::Form->new();
  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage);

    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{usingForms} !~ /^(0|1)$/)
  {
    $self->invalid("usingForms", $self->{usingForms}, "Must be a boolean value (0 or 1).");
  }

  if (length $self->{objName} == 0)
  {
    $self->missing("objName");
  }

  if (! exists $self->{data})
  {
    $self->missing("data");
  }
  elsif (! exists $self->{data}->{main})
  {
    $self->missing("main", "You must define form 'main'!");
  }
  elsif (scalar keys %{$self->{data}->{main}} == 0)
  {
    $self->missing("main", "You must define the variables that make up your object.");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

sub AUTOLOAD
{
  my $self = shift;
  my $type = ref($self);
  my $i=0;
  my $result = "";
  while (my @info=caller($i++))
  {
    $result .= "$info[2] - $info[3]: $info[4]\n";
  }
  if (!ref($self))
  {
    die "$self is not an object\nCalled from:\n$result";
  }
  my $name = $AUTOLOAD;

  # make sure we don't emulate the DESTROY() method.
  return if $name =~ /::DESTROY$/;

  $name =~ s/.*://; # strip fully-qualified portion
  if (exists $self->{$name})
  {
    if (@_)
    {
      my $value = shift;

      # update the data hashes, if they contain this element.
      foreach my $form (keys %{$self->{data}})
      {
        if (exists $self->{data}->{$form}->{$name})
        {
          $self->{data}->{$form}->{$name}->{-Value} = $value;
        }
      }
      return $self->{$name} = $value;
    }
    else
    {
      return $self->{$name};
    }
  }
  die "$result\nCan't access `$name' field in object of class $type";
}

=item scalar get("name")

 requires: "name" - name of the entry to work with.
 returns: data->{main}->{"name"}->{-Value}

 die's if the entry does not exist in data->{main}.

 Calls var("name") to do the actual work.
 Ex: print $obj->get("uname");

=cut
sub get
{
  my $self = shift;

  return $self->var(@_);
}

=item scalar set("name", "value")
=item scalar set(name => "value")

 requires: "name" - name of the entry to work with.
           "value" - value to assign to data->{main}->{"name"}->{-Value}
 returns: "value"

 Calls var("name", "value") to do the actual work.

 Ex: $obj->set("uname", "jdoe");
 or: $obj->set(uname => "jdoe");

=cut
sub set
{
  my $self = shift;

  if (scalar @_ != 2)
  {
    die "set: you must specify the name and value attributes!";
  }

  return $self->var(@_);
}

=item scalar var("name", "value")
=item scalar var(name => "value")

 requires: "name" - name of the entry to work with.
 optional: "value" - value to assign to data->{main}->{"name"}->{-Value}

 Looks up the specified entry in the data->{main} form and returns
 it's -Value entry.

 If value is specified, then we set the entry in data->{main} and
 any other forms defined that have it specified and then return the
 new -Value.

 We die with an error message if the entry does not exist in
 data->{main}.

 Examples:  $obj->var("uname");  # would get the value of "uname".
 $obj->var("uname", "jdoe"); # would set data->{main}->{uname}->{-Value} = "jdoe";

=cut
sub var
{
  my $self = shift;
  my $name = shift;
  my $i=0;
  my $result = "";
  while (my @info=caller($i++))
  {
    $result .= "$info[2] - $info[3]: $info[4]\n";
  }

  if (length $name == 0 || ! defined $name)
  {
    die "$result\nname not specified!";
  }

  if (exists $self->{data}->{main}->{$name})
  {
    if (@_)
    {
      my $value = shift;

      # update the data hashes, if they contain this element.
      foreach my $form (keys %{$self->{data}})
      {
        next if ($form eq "main");
        if (exists $self->{data}->{$form}->{$name})
        {
          $self->{data}->{$form}->{$name}->{-Value} = $value;
        }
      }

      $self->{$name} = $value if ($self->{backwardsCompatible});  # populate the $self-> entry.

      return $self->{data}->{main}->{$name}->{-Value} = $value;
    }
    else
    {
      return $self->{data}->{main}->{$name}->{-Value};
    }
  }
  die "$result\nCan't access `$name' field.";
}

=item bool objsEqual(a,b)

 returns 1 if the 2 objects are the same (content wise),
 0 otherwise.

=cut
sub objsEqual
{
  my $a = shift;
  my $b = shift;

  if (ref($a) ne ref($b))
  {
    die "objsEqual: a = '" . ref($a) . "' NOT EQUAL b = '" . ref($b) . "'";
  }

  # only compare those elements defined in the data->{main} hash.
  # first we walk a then b, to make sure that they both have the same number of elements, etc.

  foreach my $key (keys %{$a->{data}->{main}})
  {
    return 0 if (! exists $b->{data}->{main}->{$key});

    # skip over those entries that are not actual data elements of this object.
    next if (! exists $b->{data}->{main}->{$key}->{-DataType});

    if ($a->{data}->{main}->{$key}->{-DataType} =~ /^(string)$/)
    {
      return 0 if ($a->{data}->{main}->{$key}->{-Value} ne $b->{data}->{main}->{$key}->{-Value});
    }
    elsif ($a->{data}->{main}->{$key}->{-DataType} =~ /^(int|float)$/)
    {
      return 0 if ($a->{data}->{main}->{$key}->{-Value} != $b->{data}->{main}->{$key}->{-Value});
    }
  }

  foreach my $key (keys %{$b->{data}->{main}})
  {
    return 0 if (! exists $a->{data}->{main}->{$key});

    # skip over those entries that are not actual data elements of this object.
    next if (! exists $a->{data}->{main}->{$key}->{-DataType});

    if ($b->{data}->{main}->{$key}->{-DataType} =~ /^(string)$/)
    {
      return 0 if ($a->{data}->{main}->{$key}->{-Value} ne $b->{data}->{main}->{$key}->{-Value});
    }
    elsif ($b->{data}->{main}->{$key}->{-DataType} =~ /^(int|float)$/)
    {
      return 0 if ($a->{data}->{main}->{$key}->{-Value} != $b->{data}->{main}->{$key}->{-Value});
    }
  }

  return 1;  # assume they are equal if we get this far.
}

=item bool objsNotEqual(a,b)

 returns 0 if the 2 objects are the same (content wise),
 1 otherwise.

=cut
sub objsNotEqual
{
  my $a = shift;
  my $b = shift;

  if (ref($a) ne ref($b))
  {
    die "objsEqual: a = '" . ref($a) . "' NOT EQUAL b = '" . ref($b) . "'";
  }

  # take advantage of the == overload to do the brunt of our work.
  return (!($a == $b));
}

=item bool populate()

 Always populates the data->{main} structure.

 Only those entries that have -DataType defined in them will be updated.

 If usingForms = 1,

   Populates the other forms defined in data.

 End If

 returns: true/false indicating isValid() result.

 If isValid() returned false and usingForms = 1, then the formObj will
 be updated with the invalid and missing entries so that your form can
 quickly be re-displayed without having to manually update the formObj
 yourself.

=cut
sub populate
{
  my $self = shift;
  my %args = ( @_ );

  # update those entries in the data->{main} hash.
  # also for backwards compatibility, until the apps can be updated, create the $self-> entry so $obj->{var} works.
  foreach my $name (keys %{$self->{data}->{main}})
  {
    # skip over those entries that are not actual data elements of this object.
    next if (! exists $self->{data}->{main}->{$name}->{-DataType});

    # handle the case where the user is using main as their form and this is a checkbox entry.
    if (exists $self->{data}->{main}->{$name}->{-Type} && $self->{data}->{main}->{$name}->{-Type} eq "checkbox")
    {
      if (exists $args{$name})
      {
        $self->{data}->{main}->{$name}->{checked} = 1;
      }
      else
      {
        $self->{data}->{main}->{$name}->{checked} = 0;
      }
      $self->{$name} = $self->{data}->{main}->{$name}->{-Value} if ($self->{backwardsCompatible});  # use the default value, not what the user specified.
    }
    elsif (exists $args{$name})
    {
      $self->{data}->{main}->{$name}->{-Value} = $args{$name};
      $self->{$name} = $args{$name} if ($self->{backwardsCompatible});
    }
  }

  if ($self->{usingForms})
  {
    # update the data profiles.
    foreach my $profile (keys %{$self->{data}})
    {
      next if ($profile eq "main");  # skip the main form since we already populated it.
      foreach my $name (keys %{$self->{data}->{$profile}})
      {
        # skip over those entries that are not actual data elements of this object.
        next if ((! exists $self->{data}->{$profile}->{$name}->{-DataType}) && (! exists $self->{data}->{main}->{$name}->{-DataType}));

        if ($self->{data}->{$profile}->{$name}->{-Type} eq "checkbox")
        {
          if (exists $args{$name})
          {
            $self->{data}->{$profile}->{$name}->{checked} = 1;
          }
          else
          {
            $self->{data}->{$profile}->{$name}->{checked} = 0;
          }
        }
        elsif (exists $args{$name})
        {
          $self->{data}->{$profile}->{$name}->{-Value} = $args{$name};
        }
      }
    }
  }

  # do validation
  my $result = $self->isValid;

  if (!$result && $self->{usingForms})
  {
    $self->updateFormObj();
  }

  return $result;
}

=item void updateFormObj(void)

 Migrate any invalid and missing entries from the current object, into
 the formObj, but only if usingForms = 1.

=cut
sub updateFormObj
{
  my $self = shift;

  # update the formObj with the invalid & missing entries.
  foreach my $entry ($self->getInvalid())
  {
    # add to the formObj invalid structure.
    $self->formObj->invalid($entry, $self->getInvalidEntry($entry), $self->getExtraInfoEntry($entry));
    # clear the formObj valid entry.
    $self->formObj->deleteErrorsEntry("valid", $entry);
  }
  foreach my $entry ($self->getMissing())
  {
    # add to the formObj missing structure.
    $self->formObj->missing($entry, $self->getExtraInfoEntry($entry));
    # clear the formObj valid entry.
    $self->formObj->deleteErrorsEntry("valid", $entry);
  }
}

=item scalar lookupSelectName(form =>, item =>, value =>)

 Looks up the name from the specified form value for select/multi-select
 form items.  This allows you to not have to constantly query the
 database to resolve values to names.

 optional:
   form - name of form to work with - defaults to 'main'.
   value - value to lookup.  If not specified, uses the current
     value of -Value to determine the value to lookup.  If
     item is a multi-select, then value must be specified otherwise
     only the first entry in -Value will be looked up.

 required:
   item - name of the form item to work with.

 returns undef if the specified form doesn't exist or the
   specified form item is not a select/multi-select item and errors out.
   Otherwise, returns the name if found or undef if not found.

 example:

   form = 'main', item = 'color', -Value = 'blue'

   data for color =
   { names => [ "Red", "Blue", "Green" ],
     values => [ "red", "blue", "green" ] }

   $obj->lookupSelectName(form => 'main', item => 'color')

   would return "Blue".

   $obj->lookupSelectName(item => 'color', value => 'green')

   would return "Green".

   $obj->lookupSelectName(item => 'color', value => 'purple')

   would return 'undef'.

=cut
sub lookupSelectName
{
  my $self = shift;
  my %args = ( form => 'main', @_ );
  my $form = $args{form};
  my $item = $args{item};

  if (length $form == 0)
  {
    $self->missing("form");
    return undef;
  }
  if (!exists $self->data->{$form})
  {
    $self->invalid("form", $form, "does not exist in data!");
    return undef;
  }
  if (length $item == 0)
  {
    $self->missing("item");
    return undef;
  }
  if (!exists $self->data->{$form}->{$item})
  {
    $self->invalid("item", $item, "does not exist in form = '$form'!");
    return undef;
  }
  my $itemType = $self->data->{$form}->{$item}->{-Type};
  if ($itemType !~ /^(multi-)?select$/)
  {
    $self->invalid("item", $item, "-Type = '$itemType' is not a select or multi-select!");
    return undef;
  }

  # now determine if they gave us a value or if we need to use the value of -Value.
  my $Value = $self->get($item);
  my $value = (exists $args{value} ? $args{value} : (ref ($Value) eq "ARRAY" ? $Value->[0] : $Value));

  # now we loop over the options and find the index of the one that matches the specified value.
  for (my $i=0; $i < scalar @{$self->data->{$form}->{$item}->{-Options}->{values}}; $i++)
  {
    if ($self->data->{$form}->{$item}->{-Options}->{values}->[$i] =~ /^($value)$/)
    {
      return $self->data->{$form}->{$item}->{-Options}->{names}->[$i];
    }
  }

  return undef;
}

=item string printXML(fullXML bool, indent int)

 This routine returns an XML snippet for this object for use in such
 things as exporting data

 fullXML - defaults to 0 (false).  If true, we output the xml header tag.
 indent - defaults to 2.  Specifies how many spaces in our top level tag
          needs to be.

=cut
sub printXML
{
  my $self = shift;
  my $result = "";
  my %args = ( fullXML => 0, indent => 2, @_ );
  my $full = $args{fullXML};
  my $indent = ($full ? 0 : $args{indent});  # start at 0 indentation if we are making a full XML document.
  my $spaceStr = " " x $indent;

  if ($full)
  {
    $result .= "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>";
  }
  $result .= $spaceStr . "<$self->{objName}>\n";
  foreach my $name (keys %{$self->{data}->{main}})
  {
    # skip over those entries that are not actual data elements of this object.
    next if (! exists $self->{data}->{main}->{$name}->{-DataType});

    # skip the non-data parts of the object.
    next if ($self->{data}->{main}->{$name}->{-DataType} !~ /^(string|int|float)$/);
    $result .= $spaceStr . "  <$name>$self->{data}->{main}->{$name}->{-Value}</$name>\n";
  }
  $result .= $spaceStr . "</$self->{objName}>\n";

  return $result;
}

=item scalar printStr()

 returns the printable version of this object.

=cut
sub printStr
{
  my $self = shift;
  my $out = "Portal::Objects::$self->{objName} : ";

  foreach my $name (keys %{$self->{data}->{main}})
  {
    # skip over those entries that are not actual data elements of this object.
    next if (! exists $self->{data}->{main}->{$name}->{-DataType});

    # skip the non-data parts of the object.
    next if ($self->{data}->{main}->{$name}->{-DataType} !~ /^(string|int|float)$/);

    $out .= ", " if ($out =~ /=/);  # output a , if we have an = in the string.
    $out .= "$name = '" . $self->{data}->{main}->{$name}->{-Value} . "'";
  }

  $out .= "\n";

  return $out;
}

=item string printHTML(full => bool)

 This routine returns a string of html that represents this object.
 If full => 1 then a complete HTML Document will be returned.

=cut
sub printHTML
{
  my $self = shift;
  my $result = "";
  my %args = ( full => 0, @_ );
  my $full = $args{full};
  my $doc = HTMLObject::Base->new();

  $result .= "<p>\n";
  $result .= "  <b>Contents of " . $doc->formEncodeString($self->{objName}) . "</b><br /><br />\n";
  foreach my $name (keys %{$self->{data}->{main}})
  {
    # skip over those entries that are not actual data elements of this object.
    next if (! exists $self->{data}->{main}->{$name}->{-DataType});

    # skip the non-data parts of the object.
    next if ($self->{data}->{main}->{$name}->{-DataType} !~ /^(string|int|float)$/);

    $result .= "  <b>" . $doc->formEncodeString($name) . "</b>: " . $doc->formEncodeString(string => $self->{data}->{main}->{$name}->{-Value}, sequence => "formatting") . "<br />\n";
  }
  $result .= "</p>\n";
  if ($full)
  {
    $doc->setTitle($doc->formEncodeString($self->{objName}));
    $doc->print($result);
    $result = $doc->display(print => 0);
  }
  return $result;
}

=item %form printForm(form => "main", action => "", method => "post", app =>, state =>, command =>, template => "")

  The form specified by form will be generated using the
  template, data, profile structures in the object.
  Defaults to "main" if not specified.

  If you specify app, state, command, then they will be updated
  in the data structure before the form is generated.

  You can specify any other arguments that the generate() method
  accepts and they will be passed through to the generate() method.

  If you need to name the form differently than the form being worked
  with, specify name => "newname".

  The resulting html will have #LANG=x# substitutions done to make sure
  that any strings in the template or -Label's you wanted languified
  get done.

  The following sections are languified for you:

      body
      javascript

  If you need to do this yourself, see the langSubstitute() method
  provided by Portal::Base.

=cut
sub printForm
{
  my $self = shift;
  my %args = ( form => "main", @_ );
  my $formName = $args{form};
  my %return = ();

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->updateFormObj();
    $self->clearErrors("all");
  }

  if (!exists $self->{data}->{$formName})
  {
    $self->invalid("form", $formName, "form doesn't exist!");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return %return;
  }

  foreach my $arg (qw( app state command ))
  {
    if (exists $args{$arg})
    {
      $self->{data}->{$formName}->{$arg}->{-Value} = $args{$arg};
      delete $args{$arg};
    }
  }

  # take into account the order array.
  $args{order} = (exists $self->{order}->{$formName} ? $self->{order}->{$formName} : []);

  # figure out what template to use.
  my $template = (exists $args{template} ? $args{template} : $self->{template}->{$formName});

  %return = $self->{formObj}->generate(
    data => $self->{data}->{$formName},
    name => $formName, action => $self->{action},
    method => $self->{method}, profile => $self->{profile}->{$formName},
    javascript => $self->{javascript}->{$formName}, %args, template => $template);

  # fixup the body and javascript output, replacing all #LANG=x# entries with the mapped version of x.
  $return{body} = $self->langSubstitute($return{body});
  $return{javascript} = $self->langSubstitute($return{javascript});

  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage());
    return undef;
  }
  # make sure there is nothing left over in the formObj that can mess us up if this
  # formObj is being used to display another form that may have the same names
  # but those form items are not actually invalid.
  $self->formObj->clearErrors("all");

  return %return;
}

=item boolean validateForm(input => \%input, form => "main")

 Defaults to "main" if form not specified.

 If input is not specified, then we will attempt to use
 $self->{data}->{form} and validate against that, where
 form is the formName specified.

 Returns 1 if form is valid, 0 otherwise.

=cut
sub validateForm
{
  my $self = shift;
  my %args = ( input => undef, form => "main", @_ );
  my $formName = $args{form};

  if (!exists $self->{data}->{$formName})
  {
    $self->invalid("formName", $formName, "form does not exist!");
  }
  elsif (! defined $args{input})
  {
    $args{input} = $self->convertDataHash($self->{data}->{$formName});
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  my $result = $self->{formObj}->validate(input => $args{input}, profile => $self->{profile}->{$formName},
    data => $self->{data}->{$formName} );
  if ($self->{formObj}->error)
  {
    $self->error($self->{formObj}->errorMessage);
    return 0;
  }

  return $result;
}

=item hashref convertDataHash(hashref)

 Converts the passed in hashref containing a Form data structure,
 to be a hash of name => value pairs.

 Returns the hashref to the consolidated hash.

=cut
sub convertDataHash
{
  my $self = shift;
  my $data = shift;
  my $input = {};

  foreach my $key (keys %{$data})
  {
    $input->{$key} = $data->{$key}->{-Value};
  }

  return $input;
}

=back

=cut

1;
__END__

=head1 QUESTIONS

=item Q) how do i know if a checkbox has been checked by using the object
internals:

A) $self->{data}->{$form}->{checkboxVar}->{checked} == 1

=item Q) How do I set a checkbox to be checked by default in my form?

A) add checked => 1

=item Q) What is the -Value used for when using a checkbox?

A) This is the value that will be output in the form as part of the
checkbox in the value="" attribute.  Also when the box is checked, this
value is what will be assigned to the input variable and will be placed
into the -Value for the checkbox during populate.

=head1 NOTE

 All non-data fields are accessible by:
 $obj->variable($newValue); or $value = $obj->variable;
 Ex: $obj->langObj;  # would return the langObj to work with.

 All data fields must be accessed via the get()/set() accessor
 methods.
 Ex: $obj->get("variable"); or $obj->set(variable => "new value");

 $obj->get("uname");  or $obj->set(uname => "jdoe");

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3), HTMLObject::Form(3)

=cut
