# CompanyObject.pm - The Object Class that provides a CompanyObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::CompanyObject;
use strict;
use Portal::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.07';

=head1 NAME

CompanyObject - Object used to build a CompanyObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::CompanyObject;
  my $obj = Portal::Objects::CompanyObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

CompanyObject is a CompanyObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(id, parentId, code, name, address1, address2, city, state,
zip, phone, fax, email, comment, language, tz, timeFormat,
active, country, lastEdited, url, logoUrl, billingMethod)

 Creates a new instance of the Portal::Objects::CompanyObject module.
 See Portal::Base(3) for a listing of required arguments.
 
 requires:
   id
   parentId
   code
   name
   address1
   address2
   city
   state
   zip
   phone
   fax
   email
   comment
   language
   tz
   timeFormat
   active
   country
   lastEdited
   url
   logoUrl
   billingMethod

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( id => -1, parentId => -1, code => "", name => "", address1 => "", address2 => "", city => "", state => "", zip => "", phone => "", fax => "",
               email => "", comment => "", language => 'en', tz => "CST", active => 1, timeFormat => 12, country => "US", lastEdited => "now()", url => "", logoUrl => "", billingMethod => "n", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{id} = $args{id};
  $self->{parentId} = $args{parentId};
  $self->{code} = $args{code};
  $self->{name} = $args{name};
  $self->{address1} = $args{address1};
  $self->{address2} = $args{address2};
  $self->{city} = $args{city};
  $self->{state} = $args{state};
  $self->{zip} = $args{zip};
  $self->{phone} = $args{phone};
  $self->{fax} = $args{fax};
  $self->{email} = $args{email};
  $self->{comment} = $args{comment};
  $self->{language} = $args{language};
  $self->{active} = $args{active};
  $self->{tz} = $args{tz};
  $self->{timeFormat} = $args{timeFormat};
  $self->{country} = $args{country};
  $self->{lastEdited} = $args{lastEdited};
  $self->{url} = $args{url};
  $self->{logoUrl} = $args{logoUrl};
  $self->{billingMethod} = $args{billingMethod};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{name} == 0)
  {
    $self->missing("name");
  }
  if (length $self->{code} == 0)
  {
    $self->missing("code");
  }
  if ($self->{parentId} < -1 || $self->{parentId} == 0)
  {
    $self->invalid("parentId", "$self->{parentId}");
  }
  if (length $self->{address1} == 0)
  {
    $self->missing("address1");
  }
  if (length $self->{city} == 0)
  {
    $self->missing("city");
  }
  if (not exists $self->variables->countryCodes->{$self->{country}})
  {
    $self->invalid("country", "$self->{country}");
  }
  my $country = $self->{country};
  if ($country eq "US")  # do US specific checks.
  {
    if (not exists $self->variables->states->{$self->{state}})
    {
      $self->invalid("state", "$self->{state}");
    }
    if ($self->{zip} !~ /^\d{5}(-\d{4})?$/)
    {
      $self->invalid("zip", "$self->{zip} must be 5 digits long or 5 digits - 4 digits!");
    }
    if ($self->{phone} !~ /^\d{3}(-|.)\d{3}(-|.)\d{4}$/)
    {
      $self->invalid("phone", "$self->{phone} must be 12 digits long including -'s!");
    }
    if (length $self->{fax} > 0 && $self->{fax} !~ /^\d{3}(-|.)\d{3}(-|.)\d{4}$/)
    {
      $self->invalid("fax", "$self->{fax} must be 12 digits long including -'s!");
    }
  }
  else
  {
    if (length $self->{state} == 0)
    {
      $self->invalid("state", "$self->{state}");
    }
    if (length $self->{zip} == 0)
    {
      $self->invalid("zip", "$self->{zip}");
    }
    if (length $self->{phone} == 0)
    {
      $self->invalid("phone", "$self->{phone}");
    }
  }
  if (length $self->{email} == 0)
  {
    $self->missing("email");
  }
  if ($self->{email} !~ /.+\@.+\..+/)
  {
    $self->invalid("email", "$self->{email}");
  }
  my $doc = HTMLObject::Base->new;
  if (length $self->{language} == 0)
  {
    $self->missing("language");
  }
  if (length $doc->lookupLanguageName(code => $self->{language}) == 0)
  {
    $self->invalid("language", "$self->{language}");
  }
  if ($self->{active} !~ /^(1|0)$/)
  {
    $self->invalid("active", "$self->{active}");
  }
  if (not exists $self->variables->timeZones->{$self->{tz}})
  {
    $self->invalid("tz", "$self->{tz}");
  }
  if (not exists $self->{variables}->{timeFormats}->{$self->{timeFormat}})
  {
    $self->invalid("timeFormat", "$self->{timeFormat}");
  }
  if (length $self->{lastEdited} == 0)
  {
    $self->missing("lastEdited");
  }
  if (length $self->{url} == 0)
  {
    $self->missing("url");
  }
  if (length $self->{logoUrl} == 0)
  {
    $self->missing("logoUrl");
  }
  if ($self->{billingMethod} !~ /^(n|c|i)$/)
  {
    $self->invalid("billingMethod", "$self->{billingMethod}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported Variables

  id         - int
  parentId   - int (-1 = parent, >0 points to parent)
  code       - text (uniqueness value)
  name       - text (can be duplicated in system)
  address1   - text
  address2   - text
  city       - text
  state      - text
  zip        - text (5 digits '-' 4 digits)
  phone      - text (13)
  fax        - text (13)
  email      - text
  comment    - text
  language   - text
  active     - bool
  tz         - text
  timeFormat - int (12 = AM/PM, 24 = Military Time)
  country    - text (2)
  lastEdited - time stamp
  url        - text
  logoUrl    - text
  billingMethod - char (n = none, c = credit card, i = invoice)
                  If billingMethod = n, then no for fee applications
                  can be assigned to this company.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Auth(3), Portal::Base(3)

=cut
