# LicenseObject.pm - The Object Class that provides a LicenseObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::LicenseObject;
use strict;
use Portal::Base;
use Portal::Data::Variables;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

=head1 NAME

LicenseObject - Object used to build a LicenseObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::LicenseObject;
  my $obj = Portal::Objects::LicenseObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

LicenseObject is a LicenseObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(appId, companyId, numTotal, numFree, cost, unit)

 Creates a new instance of the Portal::Objects::LicenseObject module.
 See Portal::Base(3) for a listing of required arguments.

 requires:
  appId
  companyId
  numTotal
  numFree
  cost
  unit

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( appId => -1, companyId => -1, numTotal => 0, numFree => 0, cost => "0.00", unit => 0, @_ );
  
  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{variables} = Portal::Data::Variables->new(langObj => $self->{langObj});
  $self->{appId} = $args{appId};
  $self->{companyId} = $args{companyId};
  $self->{numTotal} = $args{numTotal};
  $self->{numFree} = $args{numFree};
  $self->{cost} = $args{cost};
  $self->{unit} = $args{unit};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{appId} < 0)
  {
    $self->invalid("appId", "$self->{appId}");
  }
  if ($self->{companyId} < 0)
  {
    $self->invalid("companyId", "$self->{companyId}");
  }
  if ($self->{cost} !~ /^\d+\.\d{2}$/)
  {
    $self->invalid("cost", "$self->{cost}");
  }
  if (!exists $self->{variables}->{units}->{$self->{unit}})
  {
    $self->invalid("unit", "$self->{unit}");
  }
  if ($self->{numTotal} < 0)  # numTotal = 0 means unlimited licenses.
  {
    $self->invalid("numTotal", "$self->{numTotal}");
  }
  if ($self->{numFree} < 0)
  {
    $self->invalid("numFree", "$self->{numFree}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported VARIABLES

  appId     int
  companyId int
  numTotal  int
  numFree   int
  cost      numeric(6)
  unit      int

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Application(3), Portal::Base(3)

=cut
