# LogEntry.pm - The Object Class that provides a LogEntry Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::LogEntry;
use strict;
use Portal::Data::Config;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.05';

=head1 NAME

LogEntry - Object used to build a LogEntry Object Class.

=head1 SYNOPSIS

  use Portal::Objects::LogEntry;
  my $obj = Portal::Objects::LogEntry->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

LogEntry is a LogEntry class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(actionDate, actionTime, action, ipAddress, extraInfo, userId)

 Creates a new instance of the LogEntry module.
 See Portal::Base(3) for a listing of required arguments.

 requires: 
    actionDate - The Date. (Defaults to 01-01-2000)
    actionTime - The Time. (Defaults to 00:00:00)
    action - Integer defining the event that happened.  Application
             dependant. (Defaults to -1)
    ipAddress - The IP Address of the client machine.
                (Defaults to 127.0.0.1)
    extraInfo - Any extra text that needs to be stored.
                (Defaults to "")
    userId - Integer id of the user. (Defaults to -1)

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( actionDate => "01-01-2000", actionTime => "00:00:00", action => -1, ipAddress => "127.0.0.1", extraInfo => "", userId => -1, serverName => "", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{actionDate} = $args{actionDate};
  $self->{actionTime} = $args{actionTime};
  $self->{action} = $args{action};
  $self->{ipAddress} = $args{ipAddress};
  $self->{extraInfo} = $args{extraInfo};
  $self->{userId} = $args{userId};
  $self->{serverName} = $args{serverName};
  eval { $self->{configObj} = Portal::Data::Config->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->error("Error instantiating Portal::Data::Config->new()!<br>\nError = $@<br>\n");
    return $self;
  }
  if ($self->{configObj}->error)
  {
    $self->error($self->{configObj}->errorMessage);
    return $self;
  }

  # If they didn't specify the serverName value then use the Configuration value.
  $self->{serverName} = $self->{configObj}->myHostName if (length $self->{serverName} == 0);

  # fixup the IP Address.  Strip off the /# that Postgresql will generate.
  $self->{ipAddress} =~ s/^(.*)\/\d+$/$1/;

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  
  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{actionDate} == 0)
  {
    $self->missing("actionDate");
  }
  elsif ($self->{actionDate} !~ /^\d{1,2}(\/|-)\d{1,2}(\/|-)\d{4}$/)
  {
    $self->invalid("actionDate", "$self->{actionDate}");
  }
  if (length $self->{actionTime} == 0)
  {
    $self->missing("actionTime");
  }
  elsif ($self->{actionTime} !~ /^\d{2}:\d{2}:\d{2}$/)
  {
    $self->invalid("actionTime", "$self->{actionTime}");
  }
  if ($self->{action} < -1)
  {
    $self->invalid("action", "$self->{action}");
  }
  if (length $self->{ipAddress} == 0)
  {
    $self->missing("ipAddress");
  }
  else
  {
    if ($self->{ipAddress} !~ /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
    {
      $self->invalid("ipAddress", "$self->{ipAddress}");
    }
  }
  if ($self->{userId} !~ /^(-1|\d+)$/)
  {
    $self->invalid("userId", "$self->{userId}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__


=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Log(3), Portal::Base(3)

=cut
