# Preferences.pm - The Object Class that provides a Preferences Object
# Created by James A. Pattie, 11/07/2000.
# Last edited 07/26/2001.

# Copyright (c) 2000-2005 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::Preferences;
use strict;
use Portal::Objects::Base;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Objects::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

=head1 NAME

Preferences - Object used to build a template Object Class.

=head1 SYNOPSIS

  use Portal::Objects::Preferences;
  my $obj = Portal::Objects::Preferences->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Preferences is a template class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(methods, authObj, applicationObj, userObj, appName)

 Creates a new instance of the Portal::Objects::Preferences module.
 See Portal::Objects::Base(3) for a listing of required arguments.

 form 'main' defines the preferences that can be edited.
 form 'edit' defines the actual form items needed to edit the
   Portal's preferences.

 requires:
   methods - Portal::Methods instance
   authObj - Portal::Auth instance
   applicationObj - Portal::Application instance
   userObj - the user we are editing preferences for
   appName - the name of the App the preferences are being edited for.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( methods => undef, authObj => undef, applicationObj => undef, userObj => undef, appName => "",  @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  $self->{methods} = $args{methods};
  $self->{authObj} = $args{authObj};
  $self->{applicationObj} = $args{applicationObj};
  $self->{userObj} = $args{userObj};
  $self->{appName} = $args{appName};

  if (!$self->requiredArgsValid)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{objName} = "Preferences";
  $self->{data}->{main} =
    { "colorScheme" => { -DataType => "string", -Value => "Default", -PrefModule => "Global" },
      "dynamicContent" => { -DataType => "int", -Value => 0, -PrefModule => "Desktop" },
      "dynamicContentConfig" => { -DataType => "string", -Value => "", -PrefModule => "Desktop" },
      "displayEmptyAppGroups" => { -DataType => "int", -Value => 0, -PrefModule => "Desktop" },
    };

  $self->{usingForms} = 1;

  # the templates

  $self->{data}->{edit} =
    { "app" => { -Type => "hidden", -Value => "" },
      "state" => { -Type => "hidden", -Value => "" },
      "command" => { -Type => "hidden", -Value => "editAppPreferences" },
      "userId" => { -Type => "hidden", -Value => $self->userObj->get("id") },
      "uname" => { -Type => "hidden", -Value => $self->userObj->get("uname") },
      "appName" => { -Type => "hidden", -Value => $self->appName },
      "colorScheme" => { -Type => "select", -Label => $self->langObj->map("colorScheme"), -Value => "Default", -Options => undef },
      "dynamicContentHeader" => { -Type => "text", -Label => $self->langObj->map("dynamicContentConfig"), -Value => "", -ReadOnly => 1, -ReadOnlyMode => "text", -CreateTemplate => { -Type => "header" } },
      "dynamicContent" => { -Type => "checkbox", -Label => $self->langObj->map("dynamicContent"), -Value => "1" },
      "frameBorder" => { -Type => "select", -Label => $self->langObj->map("frameBorder"), -Value => "0", -Options => $self->frameBorderOptions() },
      "frameScrolling" => { -Type => "select", -Label => $self->langObj->map("frameScrollable"), -Value => "auto", -Options => $self->frameScrollingOptions() },
      "frameWidth" => { -Type => "text", -Label => $self->langObj->map("frameWidth"), -Value => "", size => "4", maxlength => "4" },
      "frameHeight" => { -Type => "text", -Label => $self->langObj->map("frameHeight"), -Value => "", size => "3", maxlength => "3" },
      "displayEmptyAppGroups" => { -Type => "checkbox", -Label => $self->langObj->map("displayEmptyAppGroups"), -Value => "1", checked => 0 },
      "submitForm" => { -Type => "submit", -Value => $self->langObj->map("Update") },
      "formSubmitted" => { -Type => "hidden", -Value => 1 },
    };

  $self->{order}->{edit} = [ qw(colorScheme displayEmptyAppGroups dynamicContent dynamicContentHeader frameBorder frameScrolling frameWidth frameHeight) ];

  $self->{template}->{edit} = $self->formObj->createTemplate(data => $self->{data}->{edit}, class => "edit", rowsAlternate => 0,
    header => $self->langObj->map("EditAppUserPreferences"), order => $self->{order}->{edit});
  if ($self->formObj->error)
  {
    $self->error($self->formObj->errorMessage);
    return $self;
  }

  $self->{profile}->{edit} =
    { required => [ qw(app state command uname appName colorScheme frameBorder frameScrolling frameWidth frameHeight formSubmitted) ],
      optional => [ qw(dynamicContent displayEmptyAppGroups) ],
      constraints => {},
    };

  return $self;
}

=item bool requiredArgsValid

 Checks to make sure that the required args to new() were provided
 and appear to be valid.

=cut
sub requiredArgsValid
{
  my $self = shift;

  if (!defined $self->{methods})
  {
    $self->missing("methods");
  }
  elsif (!$self->{methods}->isValid)
  {
    $self->invalid("methods", $self->{methods}->errorMessage);
  }

  if (!defined $self->{authObj})
  {
    $self->missing("authObj");
  }
  elsif (!$self->{authObj}->isValid)
  {
    $self->invalid("authObj", $self->{authObj}->errorMessage);
  }

  if (!defined $self->{applicationObj})
  {
    $self->missing("applicationObj");
  }
  elsif (!$self->{applicationObj}->isValid)
  {
    $self->invalid("applicationObj", $self->{applicationObj}->errorMessage);
  }

  if (!defined $self->{userObj})
  {
    $self->missing("userObj");
  }
  elsif (!$self->{userObj}->isValid)
  {
    $self->invalid("userObj", $self->{userObj}->errorMessage);
  }

  if (length $self->{appName} == 0)
  {
    $self->missing("appName");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # make sure our required args are valid.
  if (!$self->requiredArgsValid)
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->get("colorScheme") == 0)
  {
    $self->missing("colorScheme");
  }
  if ($self->get("dynamicContent") !~ /^(0|1)$/)
  {
    $self->invalid("dynamicContent", $self->get("dynamicContent"));
  }
  if (length $self->get("dynamicContentConfig") == 0)
  {
    $self->missing("dynamicContentConfig");
  }
  if ($self->get("displayEmptyAppGroups") !~ /^(0|1)$/)
  {
    $self->invalid("displayEmptyAppGroups", $self->get("displayEmptyAppGroups"));
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item void setupEntry(scalar)

 Takes the preference name and populates the entry in the edit form.
 If the preference uses a select box to display itself, we build
 up the -Options array and populate it as needed.

 Ex: $obj->setEntry("dynamicContent");

=cut
sub setupEntry
{
  my $self = shift;
  my $pref = shift;

  if (! exists $self->{data}->{main}->{$pref})
  {
    $self->invalid("pref", "$pref", "Does not exist in main!");
    $self->error($self->genErrorString("invalid"));
    return;
  }

  if ($pref =~ /^(dynamicContent|displayEmptyAppGroups)$/)
  {
    $self->data->{edit}->{$pref}->{checked} = $self->get($pref);
  }
  elsif ($pref eq "colorScheme")
  {
    # lookup the colorSchemes (system and user) and create the -Options hash.
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->userObj->get("id"));
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return;
    }
    my %options = ( names => [], values => [] );
    foreach my $option (@systemColorSchemes, @userColorSchemes)
    {
      push @{$options{names}}, $option;
      push @{$options{values}}, $option;
    }
    $self->data->{edit}->{$pref}->{-Options} = \%options;
  }
  elsif ($pref eq "dynamicContentConfig")
  {
    # setup the XML parser needed.
    my $parserObj;
    eval { $parserObj = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
    if ($@)
    {
      $self->error($@);
      return;
    }

    my $configObj;
    eval { $configObj = $parserObj->parse(string => $self->get($pref), module => "DynamicContentConfig", version => "1.0"); };
    if ($@)
    {
      $self->error(sprintf($self->{langObj}->map("parseFailed"), $@));
      return;
    }
    foreach my $name (keys %{$configObj->{settings}})
    {
      $self->data->{edit}->{$name}->{-Value} = $configObj->{settings}->{$name};
    }
  }
}

=item bool validateEntry(scalar)

 Takes the preference name and validates that the user input in the
 formObj appears to be valid.  Does any data assembly required and
 updates the data->{main} entry for the specified preference.

 Returns 0 on error, 1 if everything is ok.

 Ex:
 if (!$obj->validateEntry("dynamicContentConfig"))
 {
   $displayForm = 1;
 }

=cut
sub validateEntry
{
  my $self = shift;
  my $pref = shift;

  if (! exists $self->{data}->{main}->{$pref})
  {
    $self->invalid("pref", "$pref", "Does not exist in main!");
    $self->error($self->genErrorString("invalid"));
    return 0;
  }

  if ($pref =~ /^(dynamicContent|displayEmptyAppGroups)$/)
  {
    # set the preference true, if it exists in the input, since it is a checkbox.
    $self->set($pref => int $self->formObj->isEntryValid($pref));
    if ($self->get($pref) !~ /^(0|1)$/)
    {
      $self->invalid($pref, $self->get($pref));
      $self->error($self->genErrorString("invalid"));
      return 0;
    }
  }
  elsif ($pref eq "colorScheme")
  {
    # lookup the colorSchemes (system and user) and create the -Options hash.
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return 0;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->userObj->get("id"));
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return 0;
    }
    my $colorScheme = $self->formObj->getValidEntry($pref);
    if (grep($colorScheme, @systemColorSchemes) || grep($colorScheme, @userColorSchemes))
    {
      $self->set($pref => $colorScheme);
    }
    else
    {
      $self->formObj->invalid($pref, $colorScheme);
      $self->error($self->genErrorString("invalid"));
      return 0;
    }
  }
  elsif ($pref eq "dynamicContentConfig")
  {
    my $frameBorder = $self->formObj->getValidEntry("frameBorder");
    my $frameScrolling = $self->formObj->getValidEntry("frameScrolling");
    my $frameWidth = $self->formObj->getValidEntry("frameWidth");
    my $frameHeight = $self->formObj->getValidEntry("frameHeight");

    my $error = 0;
    if ($frameBorder !~ /^(0|1)$/)
    {
      $self->formObj->invalid("frameBorder", $frameBorder);
      $error = 1;
    }
    if ($frameScrolling !~ /^(yes|no|auto)$/)
    {
      $self->formObj->invalid("frameScrolling", $frameScrolling);
      $error = 1;
    }
    if ($frameWidth !~ /^(\d+%?)$/)
    {
      $self->formObj->invalid("frameWidth", $frameWidth);
      $error = 1;
    }
    if ($frameHeight !~ /^(\d+)$/)
    {
      $self->formObj->invalid("frameHeight", $frameHeight);
      $error = 1;
    }
    if ($error)
    {
      return 0;
    }

    my $configObj;
    eval { $configObj = Portal::XML::ConfigObject->new(langObj => $self->{langObj}); };
    if ($@)
    {
      $self->formObj->error($@);
      return 0;
    }
    $configObj->{version} = "1.0";
    $configObj->{module} = "DynamicContentConfig";
    $configObj->{settings}->{frameBorder} = $frameBorder;
    $configObj->{settings}->{frameScrolling} = $frameScrolling;
    $configObj->{settings}->{frameHeight} = $frameHeight;
    $configObj->{settings}->{frameWidth} = $frameWidth;

    my $configObjStr;
    eval { $configObjStr = $configObj->generateXML; };
    if ($@)
    {
      $self->formObj->error($@);
      return 0;
    }
    if (length $configObjStr == 0)
    {
      $self->formObj->error($self->{langObj}->map("noConfigXMLGenerated"));
      return 0;
    }

    $self->set($pref, $configObjStr);
  }

  return 1;
}

sub frameBorderOptions
{
  my $self = shift;
  my %options = ( names => [qw(0 1)], values => [qw(0 1)] );

  return \%options;
}

sub frameScrollingOptions
{
  my $self = shift;
  my %options = ( names => [qw(auto yes no)], values => [qw(auto yes no)] );

  return \%options;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Objects::Base(3)

=cut
