# ReplicationObject.pm - The Object Class that provides a Database ReplicationObject Object
# Created by James A. Pattie, 07/26/2001.


# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::ReplicationObject;
use strict;
use Portal::Data::Config;
use Portal::Base;
use DBIWrapper;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

=head1 NAME

ReplicationObject - Object used to build a Database Replication Object
                    Class.

=head1 SYNOPSIS

  use Portal::Objects::ReplicationObject;
  my $obj = Portal::Objects::ReplicationObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

ReplicationObject is a Database Replication Object class.  This module
will represent an entry in the replication_tb table in the portal
database.  Use this to make an entry or to contain an entry from the
table.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(tbName, dbName, dbHost, dbType, dbPort, createDBIObject)

 Creates a new instance of the ReplicationObject module.
 See Portal::Base(3) for a listing of required arguments.

 requires:
    dbType defaults to Pg, and dbPort defaults to 5432.
    tbName must be one of (company_tb, user_tb, app_tb, app_servers_tb,
      company_app_tb, user_app_tb, rights_tb, user_rights_tb,
      user_preferences_tb, dynamic_content_tb)
    If createDBIObject = 1 (it defaults to 0) then the
    DBIWrapper object is created based upon the settings given.  If it
    was successful, then dbObj will be set and will be the DBIWrapper
    Object that points to the database specified in this object.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( tbName => "", dbName => "", dbHost => "", dbType => "Pg", dbPort => "5432", createDBIObject => 0, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{tbName} = $args{tbName};
  $self->{dbName} = $args{dbName};
  $self->{dbHost} = $args{dbHost};
  $self->{dbType} = $args{dbType};
  $self->{dbPort} = $args{dbPort};
  $self->{createDBIObject} = $args{createDBIObject};
  eval { $self->{configObj} = Portal::Data::Config->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->error("Error instantiating Portal::Data::Config->new()!<br>\nError = $@<br>\n");
    return $self;
  }
  if ($self->{configObj}->error)
  {
    $self->error($self->{configObj}->errorMessage);
    return $self;
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  if ($self->{createDBIObject})
  {
    my $dbObj = DBIWrapper->new(dbType => $self->{dbType}, dbHost => $self->{dbHost}, dbName => $self->{dbName}, dbPort => $self->{dbPort},
                                dbUser => $self->{configObj}->{dbUser}, dbPasswd => $self->{configObj}->{dbPasswd});
    if ($dbObj->error)
    {
      $self->error($dbObj->errorMessage);
      return $self;
    }
    $self->{dbObj} = $dbObj;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{tbName} == 0)
  {
    $self->missing("tbName");
  }
  elsif ($self->{tbName} !~ /^(company_tb|user_tb|app_tb|app_servers_tb|company_app_tb|user_app_tb|rights_tb|user_rights_tb|user_preferences_tb|dynamic_content_tb)$/)
  {
    $self->invalid("tbName", "$self->{tbName}");
  }
  if (length $self->{dbName} == 0)
  {
    $self->missing("dbName");
  }
  if (length $self->{dbHost} == 0)
  {
    $self->missing("dbHost");
  }
  if ($self->{dbType} !~ /^(Pg|mysql)$/)
  {
    $self->invalid("dbType", "$self->{dbType}");
  }
  if ($self->{dbPort} !~ /^(\d+)$/)
  {
    $self->invalid("dbPort", "$self->{dbPort}");
  }
  if ($self->{createDBIObject} !~ /^(1|0)$/)
  {
    $self->invalid("createDBIObject", "$self->{createDBIObject}");
  }
  if ($self->{dbName} eq $self->{configObj}->{dbName} && $self->{dbHost} eq $self->{configObj}->{dbHost} && $self->{dbType} eq $self->{configObj}->{dbType} && $self->{dbPort} eq $self->{configObj}->{dbPort})
  {
    $self->invalid("replication", "You can not Replicate the Portal Database into itself!");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# print
# generate a string that represents the contents of this object.
sub print
{
  my $self = shift;
  my $result = "tbName = '$self->{tbName}', dbName = '$self->{dbName}', dbHost => '$self->{dbHost}', dbType => '$self->{dbType}', dbPort => '$self->{dbPort}'";

  return $result;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  string print(void)
    generate a string that represents the contents of this object.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Replication(3), Portal::Base(3)

=cut
