# UserPreferenceObject.pm - The Object Class that provides a UserPreferenceObject Object
# Created by James A. Pattie, 11/07/2000.
# Last edited 07/26/2001.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::UserPreferenceObject;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

=head1 NAME

UserPreferenceObject - Object used to build a UserPreference Object Class.

=head1 SYNOPSIS

  use Portal::Objects::UserPreferenceObject;
  my $obj = Portal::Objects::UserPreferenceObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

UserPreferenceObject is a UserPreference class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(id, name, value, app, module)
  
 Creates a new instance of the Portal::Objects::UserPreferenceObject object.

 id is the id of the user this preference is for.
 name is the string representing the preference (date, time, ..)
 value is the value to be stored for this preference.
 app is the name of the app that we are dealing with (Portal, App, ..)
 module describes the part of the application this preference is
        valid in.  This allows you to have the same preference string
        (date, time, etc.) in multiple sections of your application.
        Ex. Login, Desktop, Data Input, etc.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( id => 0, name => "", value => "", app => "", module => "", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{id} = $args{id};
  $self->{name} = $args{name};
  $self->{value} = $args{value};
  $self->{app} = $args{app};
  $self->{module} = $args{module};

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{id} !~ /^(\d+)$/)
  {
    $self->invalid("id", "$self->{id}");
  }
  if (length $self->{name} == 0)
  {
    $self->invalid("name", "$self->{name}");
  }
  if (length $self->{app} == 0)
  {
    $self->invalid("app", "$self->{app}");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

# print
# requires:
# optional:
# returns: string that represents the contents of this object.
sub print
{
  my $self = shift;
  my $string = "id = '$self->{id}', name = '$self->{name}', value = '$self->{value}', app = '$self->{app}',";
  $string .= " module = '$self->{module}'";

  return $string;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS (non-Inline POD)

  string print(void)
    Returns the string that represents the contents of this object.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Auth(3), Portal::Base(3)

=cut
