# WhoEntryObject.pm - The Object Class that provides a WhoEntryObject Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::Objects::WhoEntryObject;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.06';

=head1 NAME

WhoEntryObject - Object used to build a WhoEntryObject Object Class.

=head1 SYNOPSIS

  use Portal::Objects::WhoEntryObject;
  my $obj = Portal::Objects::WhoEntryObject->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

WhoEntryObject is a WhoEntryObject class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(uname, companyId, ipAddress, session, startDate,
refreshDate, skipDates)
    
 Creates a new instance of the Portal::Objects::WhoEntryObject object.
 See Portal::Base(3) for a listing of required arguments.

 requires:
   uname
   companyId
   ipAddress
   session
   startDate
   refreshDate
   skipDates

 If skipDates = 1, then don't validate
    startDate and refreshDate values.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( uname => "", companyId => -1, ipAddress => "", session => "", startDate => "", refreshDate => "", skipDates => 0, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{uname} = $args{uname};
  $self->{companyId} = $args{companyId};
  $self->{ipAddress} = $args{ipAddress};
  $self->{session} = $args{session};
  $self->{startDate} = $args{startDate};
  $self->{refreshDate} = $args{refreshDate};
  $self->{skipDates} = $args{skipDates};
  
  # strip off the /# that PostgreSQL generates for the cidr type.
  $self->{ipAddress} =~ s/^(.*)\/\d+$/$1/;
  
  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if (length $self->{uname} == 0)
  {
    $self->invalid("uname", "$self->{uname}");
  }
  if ($self->{companyId} < 0)
  {
    $self->invalid("companyId", "$self->{companyId}");
  }
  if ($self->{ipAddress} !~ /^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)
  {
    $self->invalid("ipAddress", "$self->{ipAddress}");
  }
  if (length $self->{session} != 32)
  {
    $self->invalid("session", "$self->{session}");
  }
  if ($self->{skipDates} !~ /^(1|0)$/)
  {
    $self->invalid("skipDates", "$self->{skipDates}");
  }
  if ($self->{skipDates} == 0)
  {
    if ($self->{startDate} !~ /^(\d{2}(-|\/)\d{2}(-|\/)\d{4}|\d{4}(-|\/)\d{2}(-|\/)\d{2}) \d{2}:\d{2}:\d{2}( (AM|PM))?$/)
    {
      $self->invalid("startDate", "$self->{startDate}");
    }
    if ($self->{refreshDate} !~ /^(\d{2}(-|\/)\d{2}(-|\/)\d{4}|\d{4}(-|\/)\d{2}(-|\/)\d{2}) \d{2}:\d{2}:\d{2}( (AM|PM))?$/)
    {
      $self->invalid("refreshDate", "$self->{refreshDate}");
    }
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Who(3), Portal::Base(3)

=cut
