# SOAP.pm - The Class that provides SOAP methods
# Created by James A. Pattie, 2004-07-05.

# Copyright (c) 2004, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::SOAP;
use strict;
use Portal::Base;
use Portal::Session;
use Portal::Objects::User;
use Portal::Methods;
use Portal::Data::Config;
use Portal::Auth;
use Portal::Application;
use Portal::Objects::SOAPResult;
use DBIWrapper;
use Portal::Objects::CompanyObject;
use Portal::Objects::CompanyApplicationObject;
use Portal::Objects::ApplicationObject;
use Portal::Objects::UserPreferenceObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

# global variables.

=head1 NAME

SOAP - Object used to provide SOAP methods for the Portal.

=head1 SYNOPSIS

  use Portal::SOAP;
  my $obj = Portal::SOAP->new(langObj => $langObj);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

SOAP is a class of SOAP methods.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(caller, methods, configObj)

 Creates a new instance of the Portal::SOAP module.
 See Portal::Base(3) for a listing of required arguments.

 caller - a valid User Object.
 methods - instance of the Portal::Methods module.
 configObj - Portal Config Object used to setup the Databases.

 Security checks:
 The caller must exist and be an active user.

 Each method will have to add their own security checks
 to make sure the caller has the ability to do whatever that
 method is trying to do.

 NOTE: to any developers hacking on the SOAP code, any perl
 modules you want to pass in as variables, have to have a
 use statement defined so that the server SOAP code knows
 about the module and can re-create it correctly. :(

 You can not pass around objects that have DBIWrapper
 instances within them as the DBI handle in the DBIWrapper
 object does not get recreated correctly and so blows up.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( methods => undef, configObj => undef, 'caller' => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{methods} = $args{methods};
  $self->{configObj} = $args{configObj};
  $self->{caller} = $args{caller};

  # do validation, allowing ourselves to be derived from.
  if (!$self->Portal::SOAP::isValid)
  {
    # the error is set in the isValid() method.
    $self->prefixError();
    return $self;
  }

  # do anything else you might need to do.

  # now do the initial authentication checks.
  if (!$self->authenticateCaller)
  {
    $self->prefixError();
    return $self;
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;


  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  foreach my $name (qw( methods configObj caller ))
  {
    if (!defined $self->{$name})
    {
      $self->missing($name);
    }
    elsif (!$self->{$name}->isValid)
    {
      $self->invalid($name, $self->{$name}->errorMessage);
    }
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item bool authenticateCaller()

 attempts to make sure that the caller is still a valid user in the
 Portal and is active.

=cut
sub authenticateCaller
{
  my $self = shift;

  # instantiate the database related objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    $self->cleanupDB();
    return 0;
  }

  # now validate the caller's user object exists and is equal, etc.
  my $tmpUser = $self->{authObj}->getUserInfo(uname => $self->{caller}->{uname});
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    $self->cleanupDB();
    return 0;
  }
  if (!defined $tmpUser)
  {
    $self->error("User = '$self->{caller}->{uname}' not found!");
    $self->cleanupDB();
    return 0;
  }
  if ($tmpUser != $self->{caller})
  {
    $self->error("The looked up user is different than the caller object for user = '$self->{caller}->{uname}'!");
    $self->cleanupDB();
    return 0;
  }

  if (!$self->{caller}->{active})
  {
    $self->error("The caller is not active!");
    $self->cleanupDB();
    return 0;
  }

  $self->cleanupDB();
  # if they get this far, then they must be a valid user in the system and be active.
  return 1;
}

=item void setupDB(void)

 Re-creates the portalDB DBIWrapper object and recreates the
 authObj and applicationObj objects so the database code can work.

 This needs to be called by all methods that are not called via
 new() as they are running through SOAP and will not have valid
 database objects.  The sessionObj can not be referenced once the
 new() method is called as I can not "fix it" without causing a
 locking contention. :(

 Don't forget to call the cleanupDB() method before your method
 returns.  This includes when errors occur.

=cut
sub setupDB
{
  my $self = shift;
  my $dbObj = undef;

  $dbObj = $self->{methods}->portalDBSetup(type => 'portal', configObj => $self->{configObj});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return;
  }
  if ($dbObj->error)
  {
    $self->error($dbObj->errorMessage);
    return;
  }

  $self->{portalDB} = $dbObj;

  # create a dummy sessionObj that we can pass around since the doLog method relies on the session
  # in the Auth and Application modules to work properly.
  my $sessionObj = Portal::Base->new(langObj => $self->{langObj});
  if ($sessionObj->error)
  {
    $self->error($sessionObj->errorMessage);
    $self->cleanupDB();
    return;
  }
  $sessionObj->{store}->{userObj} = $self->{caller};

  # now update the authObj and applicationObj objects.
  $self->{authObj} = Portal::Auth->new(portalDB => $self->{portalDB}, sessionObj => $sessionObj, langObj => $self->{langObj});
  $self->{applicationObj} = Portal::Application->new(portalDB => $self->{portalDB}, sessionObj => $sessionObj, langObj => $self->{langObj});

  foreach my $obj (qw( authObj applicationObj ))
  {
    if ($self->{$obj}->error)
    {
      $self->error($self->{$obj}->errorMessage);
      $self->cleanupDB();
      return;
    }
  }
}

=item void cleanupDB(void)

 removes the portalDB, authObj and applicationObj
 objects before the module is re-serialized and sent
 across the wire.

=cut
sub cleanupDB
{
  my $self = shift;

  foreach my $obj (qw( portalDB authObj applicationObj ))
  {
    if (exists $self->{$obj})
    {
      delete $self->{$obj};
    }
  }
}

=item void DELETE(void)

 Restores the saved portalDB values so the SOAP code doesn't blow
 up when re-serializing it on the callers side.

 This must be called before the method returns.

=cut
sub DELETE
{
  my $self = shift;

  # make sure the cleanupDB() method has been called.
  $self->cleanupDB();
}

=item SOAPResult cleanupCompany(companyObj, dataDir)

 takes: dataDir
 returns: SOAPResult object or undef on major error
 This routine dumps any data necessary and if a database is defined,
 destroys the database.

=cut
sub cleanupCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, dataDir => "", @_ );
  my $companyObj = $args{companyObj};
  my $dataDir = $args{dataDir};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (length $dataDir == 0 || ! -d $dataDir)
  {
    $self->invalid("dataDir", $dataDir, " - is invalid or does not exist!");
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }

  # fixup the portalDB objects for the necessary objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure the caller is a sysadmin and not in the company being removed.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin!");
  }
  if ($self->{caller}->{companyId} == $companyObj->{id})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not delete your own company!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }

  # do any cleanup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult setupCompany(companyObj)

 returns: SOAPResult object or undef on major error
 This routine proceeds to create a database for the company if
 needed and does any initialization necessary.

=cut
sub setupCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, @_ );
  my $companyObj = $args{companyObj};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }

  # fixup the portalDB objects for the necessary objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure the caller is a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }
  # do any setup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult setupUser(companyObj, userObj)

 returns: SOAPResult object or undef on major error
 This routine configures the user in the application.

=cut
sub setupUser
{
  my $self = shift;
  my %args = ( companyObj => undef, userObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }

  # fixup the portalDB objects for the necessary objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure the caller is a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }

  # do any setup necessary

  ## TODO: change this hard-coding to actually use the Preferences Object to get the elements to create and their modules!

  # Create ColorSchemes and other core preferences.
  my %userPreferences = ();
  $userPreferences{Desktop}{displayEmptyAppGroups} = 0;
  $userPreferences{Desktop}{dynamicContent} = 0;
  $userPreferences{Desktop}{dynamicContentConfig} = $self->{methods}->getConfigDefault(app => "Portal", module => "Desktop", prefName => "dynamicContentConfig_Default", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  if (!defined $userPreferences{Desktop}{dynamicContentConfig})
  {
    $self->error(sprintf($self->langObj->map("DefaultForPreferenceDoesNotExist"), "dynamicContentConfig", "Portal", "Desktop", $userObj->get("id")));
  }

  $userPreferences{Global}{colorScheme} = $self->{methods}->getConfigDefault(app => "Portal", module => "Global", prefName => "colorScheme_Default", portalDB => $self->{portalDB});
  if ($self->{methods}->error)
  {
    $self->error($self->{methods}->errorMessage);
    return undef;
  }
  if (!defined $userPreferences{Global}{colorScheme})
  {
    $self->error(sprintf($self->langObj->map("DefaultForPreferenceDoesNotExist"), "colorScheme", "Portal", "Global", $userObj->get("id")));
  }

  # create the preferences.
  foreach my $module (keys %userPreferences)
  {
    foreach my $pref (keys %{$userPreferences{$module}})
    {
      my $userPreferenceObj = Portal::Objects::UserPreferenceObject->new(id => $userObj->{id}, app => "Portal", name => "$pref", value => $userPreferences{$module}{$pref}, module => $module, langObj => $self->{langObj});
      if ($userPreferenceObj->error)
      {
        $self->error($userPreferenceObj->errorMessage);
        return undef;
      }
      my $result2 = $self->{authObj}->createUserPreference(userPreferenceObj => $userPreferenceObj);
      if ($self->{authObj}->error)
      {
        $self->error($self->{authObj}->errorMessage);
        return undef;
      }
      if ($result2 == -1)
      {
        $self->error("Somehow user $userObj->{uname} has already set preference: " . $userPreferenceObj->print . "\n");
        return undef;
      }
      elsif ($result2 == -2)
      {
        $self->error("App = 'Portal' does not exist!\nThis error should never happen!\n");
        return undef;
      }
      else
      {
        $result->{message} .= "Successfully made preference: " . $userPreferenceObj->print . "\n<br />";
      }
    }
  }

  $self->cleanupDB();
  return $result;
}

=item SOAPResult updateUser(companyObj, userObj)

 returns: SOAPResult object or undef on major error
 This routine updates the users settings in the application.
 This is mainly to synchronize any changes that were made to
 the users info in the portal_db that the application duplicates
 or relies on for user settings in its own database.

=cut
sub updateUser
{
  my $self = shift;
  my %args = ( companyObj => undef, userObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }

  # fixup the portalDB objects for the necessary objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure the caller is either the user, a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin} && $self->{caller}->{uname} != $userObj->{uname})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin, Admin or the user being updated!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }

  # do any setup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult cleanupUser(companyObj, userObj)

 returns: SOAPResult object or undef on major error
 This routine archives any info on the user and then removes
 their data from active use.

=cut
sub cleanupUser
{
  my $self = shift;
  my %args = ( companyObj => undef, userObj => undef, @_ );

  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }

  # fixup the portalDB objects for the necessary objects.
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # make sure the caller is a sysadmin or an admin in the company being cleanedup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not cleanup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }

  # do any cleanup necessary
  $self->{authObj}->deleteAllUserPreferences(userId => $userObj->{id});
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    $self->cleanupDB();
    return undef;
  }
  $result->{message} .= sprintf($self->{langObj}->map("DeletedAllUserPreferencesForUser"), $userObj->{uname}) . "<br />\n";

  # delete all the dynamicContent entries for this user.
  my $status = $self->{applicationObj}->deleteDynamicContentEntries(userId => $userObj->{id});
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    $self->cleanupDB();
    return undef;
  }
  if ($status == -2)
  {
    $result->{message} .= sprintf($self->{langObj}->map("PermissionDeniedDeleting"), $self->{langObj}->map("DynamicContentEntryForUser"), $userObj->{id}) . "<br />\n";
    $result->{result} = 0;  # signal an error did occur.
  }
  else
  {
    $result->{message} .= sprintf($self->{langObj}->map("DeletedAllDynamicContentEntriesForUser"), $userObj->{uname}) . "<br />\n";
  }

  # get all the colorSchemes for this user.
  my @colorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $userObj->{id});
  if ($self->{applicationObj}->error)
  {
    $self->error($self->{applicationObj}->errorMessage);
    $self->cleanupDB();
    return undef;
  }
  # now delete the colorSchemes that are assigned to our user, not -1.
  foreach my $colorScheme (@colorSchemes)
  {
    if ($colorScheme->{userId} == $userObj->{id})
    {
      $status = $self->{applicationObj}->deleteColorScheme(name => $colorScheme->{name}, userId => $userObj->{id});
      if ($self->{applicationObj}->error)
      {
        $self->error($self->{applicationObj}->errorMessage);
        $self->cleanupDB();
        return undef;
      }
      if ($status == -2)
      {
        $result->{message} .= sprintf($self->{langObj}->map("PermissionDeniedDeleting"), $self->{langObj}->map("ColorScheme"), $colorScheme->{name}) . "<br />\n";
        $result->{result} = 0;  # signal an error did occur.
      }
      else
      {
        $result->{message} .= sprintf($self->{langObj}->map("DeletedColorSchemeForUser"), $colorScheme->{name}, $userObj->{uname}) . "<br />\n";
      }
    }
  }

  $self->cleanupDB();
  return $result;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Base(3), Portal::Objects::SOAPResult(3)

=cut

