# ConfigObject.pm - Will pull settings from an XML config file into a format usable by the system.
# Created by James A. Pattie.  Copyright (c) 2002-2003, Xperience, Inc.

package Portal::XML::ConfigObject;

use strict;
use XML::LibXML;
use Portal::Base;
use vars qw ($AUTOLOAD @ISA @EXPORT $VERSION);

require Exporter;
@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = "1.1";

=head1 NAME

ConfigObject - The XML Portal ConfigObject Module.

=head1 SYNOPSIS

  use Portal::XML::ConfigObject;
  my $obj = Portal::XML::ConfigObject->new;

=head1 DESCRIPTION

Portal::XML::ConfigObject will contain the parsed XML file.
It provides a method to validate that it is complete and also a method
to generate a valid XML file from the data stored in the data hash.

=head1 FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(void)
 Creates a new instance of the Portal::XML::ConfigObject object.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{version} = "1.0";
  $self->{module} = "";
  $self->{settings} = {};

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.
  if ($self->{version} !~ /^(\d+\.\d+)$/)
  {
    $self->invalid("version", "$self->{version}");
  }
  if (length $self->{module} == 0)
  {
    $self->missing("module");
  }
  if (scalar keys %{$self->{settings}} == 0)
  {
    $self->missing("settings");
  }
  else
  {
    # validate settings
    #foreach my $key (%{$self->{settings}})
    #{
    #}
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

sub generateXML
{
  my $self = shift;
  my $result = "";

  if ($self->isValid())
  {
    $result .= <<"END_OF_XML";
<?xml version="1.0" encoding="ISO-8859-1"?>
<config module="$self->{module}" version="$self->{version}">
END_OF_XML
    foreach my $setting (sort keys %{$self->{settings}})
    {
      my $name = $self->encodeEntities(string => $setting);
      my $value = $self->encodeEntities(string => $self->{settings}->{$setting});
      $value =~ s/(\\\@)/\@/g;
      $result .= "  <setting name=\"$name\" value=\"$value\"/>\n";
    }
    $result .= <<"END_OF_XML";
</config>
END_OF_XML
  }
  else
  {
    $result .= "Data not valid!\n\n";
    $result .= $self->errorMessage;
    die $result;
  }

  return $result;
}

# string encodeEntities(string)
# requires: string - string to encode
# optional:
# returns: string that has been encoded.
# summary: replaces all special characters with their XML entity equivalent. " => &quot;
sub encodeEntities
{
  my $self = shift;
  my %args = ( string => "", @_ );
  my $string = $args{string};

  my @entities = ('&', '"', '<', '>', '\n');
  my %entities = ('&' => '&amp;', '"' => '&quot;', '<' => '&lt;', '>' => '&gt;', '\n' => '\\n');

  return $string if (length $string == 0);

  foreach my $entity (@entities)
  {
    $string =~ s/$entity/$entities{$entity}/g;
  }

  return $string;
}

=back

=cut

1;
__END__

=head1 Exported Functions (non-Inline POD)

  string generateXML(void)
    Creates an XML file based upon the info stored in the
    Portal::XML::ConfigObject object.  It first calls isValid() to make sure
    this is possible.  If not then we die with an informative error
    message.

=head1 VARIABLES

  version - version of the XML file parsed

  module - the module this config data is for

  settings - the hash of name/value settings
    each config entry is stored as name = value.

  An example of the config xml schema is:

<config module="Portal" version="0.6">
  <setting name="myHostName" value="MYHOSTNAME"/>
</config>

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin at pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::XML::ConfigParser(3), Portal::Base(3)

=cut
