# DBSettingsParser.pm - Will parse an XML file and return the corresponding data object.
# Created by James A. Pattie.  Copyright (c) 2002-2005, Xperience, Inc.

package Portal::XML::DBSettingsParser;

use strict;
use XML::LibXML;
use Portal::Base;
use Portal::XML::DBSettings;
use vars qw ($AUTOLOAD @ISA @EXPORT $VERSION);

require Exporter;
@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = "1.0";

=head1 NAME

DBSettingsParser - The XML Database Settings Parser Module.

=head1 SYNOPSIS

  use Portal::XML::DBSettingsParser;
  my $obj = Portal::XML::DBSettingsParser->new(langObj => $langObj);
  my $dbSettings = $obj->parse(file => "config.xml");
  # this is a DBSettings object.

=head1 DESCRIPTION

DBSettingsParser will parse XML files that have been generated to the
database settings specification.  See the Portal::XML::DBSettings man
page for the structure of the returned data.

=head1 FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(void)

 Creates a new instance of the Portal::XML::DBSettingsParser module.
 See Portal::Base(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );
  my $errStr = "Portal::XML::DBSettingsParser->new$self->{errorPhrase}";

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{dbSettingsObj} = undef;
  $self->{module} = "";
  $self->{version} = "";

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  eval { $self->{xmlObj} = XML::LibXML->new(); };
  if ($@)
  {
    die "$errStr $@\n";
  }

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item dbSettingsObj parse(file, string, module, version)

 requires:
   file - xml file to work with
   string - xml data in a string to work with.
   module - The Portal App the config file is for
   version - The version of the config file required
 returns: Portal::XML::DBSettings instance with parsed info.

 Does the actual parsing of the XML file and generates the
 resulting data object and returns it.

 If you don't specify a file to work with then you must specify the
 xml via the string argument.  If you specify both, then the string
 will take precedence.  The file must still point to a valid file.

=cut
sub parse
{
  my $self = shift;
  my %args = ( "file" => "", string => "", module => "", version => "", @_ );
  my $nodes = undef;
  my $errStr = "Portal::XML::DBSettingsParser->parse$self->{errorPhrase}";

  if (!defined $self->{xmlObj})
  {
    die "$errStr xmlObj is undefined!\n";
  }

  if (length $args{file} > 0)
  {
    if ($args{file} !~ /^(.*\.xml)$/)
    {
      die "$errStr file = '$args{file}' is not a valid file!\n";
    }
    if (! -e $args{file})
    {
      die "$errStr Can not find config file = '$args{file}'!  $!\n";
    }
    if (! -r $args{file})
    {
      die "$errStr config file = '$args{file}' is not readable!  $!\n";
    }
  }
  elsif (length $args{string} == 0)
  {
    die "$errStr You must specify either 'file' or 'string'!\n";
  }
  if (length $args{module} == 0)
  {
    die "$errStr You must specify the module this config file is for!\n";
  }
  if ($args{version} !~ /^(\d+\.\d+)$/)
  {
    die "$errStr version = '$args{version}' is invalid!\n";
  }

  $self->{module} = $args{module};
  $self->{version} = $args{version};

  $self->{dataFile} = (length $args{string} > 0 ? $args{string} : $args{file});

  if (length $args{file} > 0 && length $args{string} == 0)
  {
    eval { $self->{xmlDoc} = $self->{xmlObj}->parse_file($self->{dataFile}); };
  }
  else
  {
    eval { $self->{xmlDoc} = $self->{xmlObj}->parse_string($self->{dataFile}); };
  }
  if ($@)
  {
    die "$errStr $@\n";
  }

  # get the module
  my $module = $self->getModule;

  if ($module =~ /^($self->{module})$/)
  {
    eval { $self->{dbSettingsObj} = Portal::XML::DBSettings->new(langObj => $self->{langObj}); };
    if ($@)
    {
      die "$errStr $@\n";
    }
    if ($self->{dbSettingsObj}->error)
    {
      die "$errStr" . $self->{dbSettingsObj}->errorMessage;
    }

    # start by validating the version of the XML file.
    $self->validateVersion;

    # initiate the data structure.  Fill in any default values possible.
    $self->{dbSettingsObj}->{version} = $self->{version};
    $self->{dbSettingsObj}->{module} = $module;

    # gather the <sections>
    $self->getSections;
  }
  else
  {
    die "$errStr  Unknown config module = '$module'!\n";
  }

  return $self->{dbSettingsObj};
}

=item hash getAttributes(node)

 requires: node - XPath Node
 returns:  hash of attributes for the specified node.

=cut
sub getAttributes
{
  my $self = shift;
  my %args = ( node => undef, @_ );
  my $node = $args{node};
  my %attributes = ();
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if (!defined $node)
  {
    die "$errStr  You must specify the XPath Node to work with!\n";
  }
  if ($node->getType() != XML_ELEMENT_NODE)
  {
    die "$errStr  You did not specify an XPath Node: " . $node->getType() . "\n";
  }
  foreach my $attribute ($node->getAttributes)
  {
    my $name = $attribute->getName;
    $attributes{$name} = $attribute->getValue;
  }

  return %attributes;
}

=item array getNodes(path, context)

 required: path - XPath to search for
 optional: context - the XPath object to base the search from.
   Make sure your path is relative to it!
 returns:  array - array of nodes returned.  These are the XPath
   objects representing each node.

=cut
sub getNodes
{
  my $self = shift;
  my %args = ( path => "*", context => undef, @_ );
  my $path = $args{path};
  my $context = $args{context};
  my @nodes = ( );
  my $nodes = undef;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  if (length $path == 0)
  {
    die "$errStr  You must specify a path!\n";
  }

  if (! defined $context)
  {
    $nodes = $self->{xmlDoc}->findnodes($path);
  }
  else
  {
    $nodes = $context->findnodes($path);
  }
  if (!$nodes->isa('XML::LibXML::NodeList'))
  {
    die "$errStr  Query '$path' didn't return a nodelist: " . $nodes->getType() . "\n";
  }
  if ($nodes->size)
  {
    #print "Found " . $nodes->size . " nodes...\n";
    foreach my $node ($nodes->get_nodelist)
    {
      push @nodes, $node;
    }
  }

  return @nodes;
}

=item string getModule(void)

 returns the module value from the parent <dbSettings> tag.

=cut
sub getModule
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my @nodes = $self->getNodes(path => "/dbSettings");
  if (scalar @nodes == 0)
  {
    die "$errStr  Your XML file doesn't contain a <dbSettings> tag!\n";
  }
  if (scalar @nodes > 1)
  {
    die "$errStr  You have too many <dbSettings> tags!  You should only have one!\n";
  }
  my %attributes = $self->getAttributes(node => $nodes[0]);
  if (!exists $attributes{module})
  {
    die "$errStr  You do not have the module defined!\n";
  }

  return $attributes{module};
}

=item string getVersion(void)

 returns the version value from the parent <dbSettings> tag.

=cut
sub getVersion
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my @nodes = $self->getNodes(path => "/dbSettings");
  if (scalar @nodes == 0)
  {
    die "$errStr  Your XML file doesn't contain a <dbSettings> tag!\n";
  }
  if (scalar @nodes > 1)
  {
    die "$errStr  You have too many <dbSettings> tags!  You should only have one!\n";
  }
  my %attributes = $self->getAttributes(node => $nodes[0]);
  if (!exists $attributes{version})
  {
    die "$errStr  You do not have the version defined!\n";
  }

  return $attributes{version};
}

=item void validateVersion()

 This routine looks up the <dbSettings version=""> tag and validates
 that the version specified is the same as what we know how to work
 with.

=cut
sub validateVersion
{
  my $self = shift;
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";

  my $version = $self->getVersion;
  if ($version !~ /^($self->{version})$/)
  {
    die "$errStr  '$version' is not equal to Version '$self->{version}'!\n";
  }
}

=item void getSections(void)

 requires: nothing
 returns: nothing

=cut
sub getSections
{
  my $self = shift;
  my %args = ( @_ );
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";
  my $tag = "section";

  my @nodes = $self->getNodes(path => "/dbSettings/$tag");
  if (scalar @nodes == 0)
  {
    die "$errStr  You do not have a <$tag> tag for <dbSettings module='$self->{module}'>!\n";
  }
  foreach my $node (@nodes)
  {
    # gather all attributes of the <section> tag.
    my %attributes = $self->getAttributes(node => $node);
    my %encountered = ();
    foreach my $attribute (keys %attributes)
    {
      if (exists $encountered{$attribute})
      {
        die "$errStr  You have already defined '$attribute' in the <$tag> tag!'\n";
      }
      if ($attribute !~ /^(name)$/)
      {
        die "$errStr  '$attribute' is invalid in the <$tag> tag!'\n";
      }
      $encountered{$attribute} = 1;
      if ($attribute =~ /^(name)$/ && $attributes{name} !~ /^(config|event|dynamic-content|rights)$/)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid!  <$tag>\n";
      }
    }
    foreach my $required (qw(name))
    {
      if (!exists $encountered{$required})
      {
        die "$errStr  '$required' is required in the <$tag> tag!\n";
      }
    }

    # make sure it doesn't already exist.
    if (exists $self->{dbSettingsObj}->{sections}->{$attributes{name}})
    {
      die "$errStr  section = '$attributes{name}' already exists!\n";
    }

    # gather the groups for this section.
    my %groups = $self->getGroups(node => $node, section => $attributes{name});

    # create the section object and store it.
    $self->{dbSettingsObj}->{sections}->{$attributes{name}} = \%groups;
  }
}

=item % getGroups(node, section)

 requires: node, section - name of the section we are working with.
 returns: hash of groups

=cut
sub getGroups
{
  my $self = shift;
  my %args = ( node => undef, section => "", @_ );
  my $node = $args{node};
  my $section = $args{section};
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";
  my $tag = "group";
  my %groups = ();

  my @nodes = $self->getNodes(path => "*", context => $node);
  if (scalar @nodes == 0)
  {
    die "$errStr  No <$tag> tags defined for <section name=\"$section\">!\n";
  }
  foreach my $gnode (@nodes)
  {
    # gather all attributes of the <group> tag.
    my %attributes = $self->getAttributes(node => $gnode);
    my %encountered = ();
    foreach my $attribute (keys %attributes)
    {
      if (exists $encountered{$attribute})
      {
        die "$errStr  You have already defined '$attribute' in the <section name=\"$section\"><$tag> tag!'\n";
      }
      if ($attribute !~ /^(name)$/)
      {
        die "$errStr  '$attribute' is invalid in the <section name=\"$section\"><$tag> tag!'\n";
      }
      $encountered{$attribute} = 1;
      if ($attribute =~ /^(name)$/ && $attributes{name} !~ /^(add|update|delete)$/)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid!  <section name=\"$section\"><$tag>\n";
      }
    }
    foreach my $required (qw(name))
    {
      if (!exists $encountered{$required})
      {
        die "$errStr  '$required' is required in the <section name=\"$section\"><$tag> tag!\n";
      }
    }

    # make sure it doesn't already exist.
    if (exists $groups{$attributes{name}})
    {
      die "$errStr  group = '$attributes{name}' already exists in <section name=\"$section\">!\n";
    }

    # gather the entries for this section.
    my @entries = $self->getEntries(node => $gnode, section => $section, group => $attributes{name});

    # create the section object and store it.
    $groups{$attributes{name}} = \@entries;
  }

  return %groups;
}

=item @ getEntries(node, section, group)

 requires: node, section - name of the section we are working with.
   group - name of the group we are working with.
 returns: array of entries, where each entry is a hashref.

=cut
sub getEntries
{
  my $self = shift;
  my %args = ( node => undef, section => "", group => "", @_ );
  my $node = $args{node};
  my $section = $args{section};
  my $group = $args{group};
  my @callerArgs = caller(1);
  (my $subName = $callerArgs[3]) =~ s/^(.+)(::)([^:]+)$/$1->$3/;
  my $errStr = "$subName$self->{errorPhrase}";
  my $tag = "entry";
  my @entries = ();
  my %sectionArgs = (
    add => {
      config => qr/^(name|value)$/,
      event => qr/^(id|label)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app|arguments)$/,
      rights => qr/^(permission|description|admin|app|section)$/
    },
    delete => {
      config => qr/^(name)$/,
      event => qr/^(id)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app)$/,
      rights => qr/^(permission|app|section)$/
    },
    update => {
      config => qr/^(name|value|newValue)$/,
      event => qr/^(id|label|newValue)$/,
      'dynamic-content' => qr/^(callingApp|companyId|userId|tag|app|arguments|newValue)$/,
      rights => qr/^(permission|description|admin|app|section|newValue)$/
    }
  );
  my %requiredArgs = (
    add => {
      config => [qw(name value)],
      event => [qw(id label)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments)],
      rights => [qw(permission description admin app section)]
    },
    delete => {
      config => [qw(name)],
      event => [qw(id)],
      'dynamic-content' => [qw(callingApp companyId userId tag app)],
      rights => [qw(permission app section)]
    },
    update => {
      config => [qw(name value newValue)],
      event => [qw(id label newValue)],
      'dynamic-content' => [qw(callingApp companyId userId tag app arguments newValue)],
      rights => [qw(permission description admin app section newValue)]
    }
  );

  my @nodes = $self->getNodes(path => "*", context => $node);
  if (scalar @nodes == 0)
  {
    die "$errStr  No <$tag> tags defined for <section name=\"$section\"><group name=\"$group\">!\n";
  }
  foreach my $enode (@nodes)
  {
    # gather all attributes of the <entry> tag.
    my %attributes = $self->getAttributes(node => $enode);
    my %encountered = ();
    foreach my $attribute (keys %attributes)
    {
      if (exists $encountered{$attribute})
      {
        die "$errStr  You have already defined '$attribute' in the <section name=\"$section\"><group name=\"$group\"><$tag> tag!'\n";
      }
      if ($attribute !~ $sectionArgs{$group}{$section})
      {
        die "$errStr  '$attribute' is invalid in the <section name=\"$section\"><group name=\"$group\"><$tag> tag!'\n";
      }
      $encountered{$attribute} = $attributes{$attribute};

      # make sure we fixup any \n's in the value so that they actually get
      # replaced with a newline.
      $encountered{$attribute} =~ s/\\n/\n/mg if ($attribute eq "value" && $group eq "add");
      $encountered{$attribute} =~ s/\\n/\n/mg if ($attribute eq "newValue" && $group eq "update");

      if ($section eq "rights" && $attribute =~ /^(admin)$/ && $attributes{$attribute} !~ /^(0|1)$/)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid!  <section name=\"$section\"><group name=\"$group\"><$tag>\n";
      }
      if ($section eq "dynamic-content" && $attribute =~ /^(companyId|userId)$/ && $attributes{$attribute} != -1)
      {
        die "$errStr  '$attribute' = '$attributes{$attribute}' is invalid, must be -1!  <section name=\"$section\"><group name=\"$group\"><$tag>\n";
      }
    }
    foreach my $required (@{$requiredArgs{$group}{$section}})
    {
      if (!exists $encountered{$required})
      {
        die "$errStr  '$required' is required in the <section name=\"$section\"><group name=\"$group\"><$tag> tag!\n";
      }
    }

    if ($section eq "rights")
    {
      # gather the dependsOn tag.
      my @subNodes = $self->getNodes(path => "*", context => $enode);
      if (scalar @subNodes > 1)
      {
        die "$errStr  You can only have 0 or 1 <dependsOn> tags defined for <section name=\"$section\"><group name=\"$group\"><$tag>!\n";
      }
      foreach my $dnode (@subNodes)
      {
        # gather all attributes of the <dependsOn> tag.
        my %subAttributes = $self->getAttributes(node => $dnode);
        my %subEncountered = ();
        foreach my $subAttribute (keys %subAttributes)
        {
          if (exists $subEncountered{$subAttribute})
          {
            die "$errStr  You have already defined '$subAttribute' in the <section name=\"$section\"><group name=\"$group\"><$tag><dependsOn> tag!'\n";
          }
          if ($subAttribute !~ /^(app|section|permission)$/)
          {
            die "$errStr  '$subAttribute' is invalid in the <section name=\"$section\"><group name=\"$group\"><$tag><dependsOn> tag!'\n";
          }
          $subEncountered{$subAttribute} = $subAttributes{$subAttribute};
        }
        foreach my $required (qw(app section permission))
        {
          if (!exists $subEncountered{$required})
          {
            die "$errStr  '$required' is required in the <section name=\"$section\"><group name=\"$group\"><$tag><dependsOn> tag!\n";
          }
        }

        # create the dependsOn object and store it.
        $encountered{dependsOn} = \%subEncountered;
      }
    }

    # create the section object and store it.
    push @entries, \%encountered;
  }

  return @entries;
}

=back

=cut

1;
__END__

=head1 VARIABLES

  dbSettingsObj - DBSettings object that represents the xml file.

  module - The Portal App name the user specified to parse().

  version - The version of the dbSettings file the user specified to
    parse().

  xmlObj - The XML::LibXML object being used to parse the XML File.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin at pcxperience.com)

=head1 SEE ALSO

perl(1), Portal::XML::DBSettings(3), Portal::Base(3)

=cut
