# Help.pm - The Object Class that provides a Help Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::Data::Help;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.04';

# Help Data structure
my %Topics = (
             en => {
                     Normal => {
                               "_order_" => "Close App,_seperator_,Refresh,Help,_seperator_,User", #,_line_
                               "Close App" => "This closes the User Properties App, and brings you back to the Portal Desktop.",
                               "Refresh" => "This redisplays the page you are currently on. &nbsp;Use this to reset back to defaults if things get changed too drastically.",
                               "Help" => "This is what you are currently seeing.",
                               "User" => {
                                         "_order_" => "Change Password,Edit User Info,View User Info,Edit Preferences",
                                         "Change Password" => "This module will allow you to specify a new password to use. &nbsp;It will not go into affect until you log out and log back in. &nbsp;You must re-enter your current password before it will let you specify a new one.",
                                         "Edit User Info" => "This module will allow you to edit the modifiable properties of your account (tz, language, name, email, comment, etc.).",
                                         "View User Info" => "Allows you to see the info about your User account.&nbsp; Go <help app=\"\" content=\"Normal,User,Edit User Info\">here</help> to edit your User Info.",
                                         "Edit Preferences" => "This module will allow you to modify the preferences that the Portal uses (colorscheme, dynamicContent, etc.).",
                                         },
                               },
                     System => {
                               "_order_" => "This should never be reached",
                               "This should never be reached" => "The User Properties app always runs in Normal mode!<br><br>Go <help app=\"\" content=\"Normal,Help\">here</help> to return to the Normal Help System.",
                               },
                   },
             );

=head1 NAME

Help - Object used to build a Help Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::Data::Help;
  my $obj = Portal::UserProperties::Data::Help->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Help is a Help class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(lang)

 Creates a new instance of the Portal::UserProperties::Data::Help object.
 See Portal::Base(3) for a listing of required arguments.

 requires:
   lang - language code to work with.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( lang => "en", @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{lang} = $args{lang};  # we will assume this language is valid as the Language module will have already validated it for us.
  $self->{Topics} = \%Topics;   # store the help data in the class.
  if (!exists $self->{Topics}->{$self->{lang}})
  {
    $self->{lang} = "en";  # This must always exist!
  }

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=back

=cut

1;
__END__

=head1 Exported FUNCTIONS

  Topics - Help topics hash.  The language code is the first index,
           followed by Normal and System.  They provide the main indexes
           for the Normal and System modes.  Under them are the topics
           of interest for the 2 modes.

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::Help(3), Portal::Base(3)

=cut
