# Menu.pm - The Object Class that provides the Menus for UserProperties App
# Created by James A. Pattie, 02/12/2003.

# Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::Objects::Menu;
use strict;
use Portal::Base;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::Base Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.03';

=head1 NAME

Menu - Object used to build a UserOpbject Menu Class.

=head1 SYNOPSIS

  use Portal::UserProperties::Objects::Menu;
  my $obj = Portal::UserProperties::Objects::Menu->new(methods,
                sessionObj, langObj, userObj);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

Menu is a menu class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new(methods, sessionObj, langObj, userObj, portalSession, urls)

 Creates a new instance of the Portal::UserProperties::Objects::Menu
 object.
 See Portal::Base(3) for a listing of required arguments.

 requires: methods - methods object,
           sessionObj - sessionObj for app,
           userObj - userObj for user
           urls - the hash of urls defined for this application,
           portalSession - Portal SessionObj,

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( methods => undef, sessionObj => undef, urls => {}, portalSession => undef, @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{methods} = $args{methods};
  $self->{sessionObj} = $args{sessionObj};
  $self->{urls} = $args{urls};
  $self->{url} = $self->{urls}->{url};
  $self->{portalSession} = $args{portalSession};
  $self->{app} = (split /::/, ref $self)[1];

  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  $self->SUPER::isValid();

  # validate our parameters.
  if (! defined $self->{methods})
  {
    $self->missing("methods");
  }
  if (! defined $self->{sessionObj})
  {
    $self->missing("sessionObj");
  }
  if (! defined $self->{urls})
  {
    $self->missing("urls");
  }
  elsif (scalar keys %{$self->{urls}} == 0)
  {
    $self->missing("urls");
  }
  if (length $self->{url} == 0)
  {
    $self->missing("url");
  }
  if (! defined $self->{portalSession})
  {
    $self->missing("portalSession");
  }

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item % updateEntries(location, input, left, right, center)

 requires: location - (menubar, sidemenu), input - input hash,
   left, right, center (arrays of menu items)
 returns:  hash with members:
   title - title of document to be set
   left - arrayref of left menu items
   right - arrayref of right menu items
   center - arrayref of center menu items

 If location = 'sidemenu', then only the left menu array will be
   populated or worked with.

 Updates the title and menu array entries that will be processed
 by the printMenu() method in Portal::AppState to actually
 generate the menu.

=cut
sub updateEntries
{
  my $self = shift;
  my %args = ( location => "menubar", input => undef, left => [], right => [], center => [], @_ );
  my $location = $args{location};
  my $state = $args{input}->{state};
  my $command = $args{input}->{command};
  my %result = ( title => "", left => $args{left}, right => $args{right}, center => $args{center} );

  if ($location !~ /^(menubar|sidemenu)$/)
  {
    $self->invalid("location", $location);
  }
  if ($state !~ /^(Main|User)$/)
  {
    $self->invalid("state", $state);
  }
  if ($command !~ /^(.+)$/)
  {
    $self->invalid("command", $command);
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return %result;
  }

  my $url = $self->url;

  # make the customized refresh and help urls.
  my %refreshArgs = (app => $self->{app}, state => $state, command => $command);
  my $refreshUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => \%refreshArgs);
  my $helpUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => { app => "Portal", state => "Help", command => "display", helpApp => $self->{app}, content => "Normal" . ($state ne "Main" ? ",$state" : "") });

  # make the other urls needed by the sidemenu.
  my $mainMenuUrl = $self->{urls}->{mainMenuUrl};
  my $changePasswordUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "changePassword" });
  my $userPreferencesUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "preferences" });
  my $userInfoUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "display" });
  my $editUserInfoUrl = $self->{methods}->urlBuilder(baseUrl => $url, arguments => { app => $self->{app}, state => "User", command => "editUser" });

  my $changePasswordPhrase = $self->langObj->map("changePassword");
  my $userPreferencesPhrase = $self->langObj->map("userPreferences");
  my $userInfoPhrase = $self->langObj->map("userInfo");
  my $editUserInfoPhrase = $self->langObj->map("EditUserInfo");
  my $editPhrase = $self->langObj->map("edit");

  my $mainMenu = $self->langObj->map("mainMenu");

  # build up the menuItems arrays
  my @itemsLeft = @{$args{left}};
  my @itemsRight = @{$args{right}};
  my @itemsCenter = @{$args{center}};

  if ($location eq "menubar")
  {
    # Update the Help menu item.
    my $onclick = $itemsLeft[2]->get("onClick");
    $onclick =~ s/^(showHelp\('[^']+', ')([^']+)(', \d+, \d+\); return false;)$/$1$helpUrl$3/;
    $itemsLeft[2]->set(onClick => $onclick);
    $itemsLeft[2]->set(url => $helpUrl);

    # Update the Refresh menu item.
    $itemsLeft[4]->set(url => $refreshUrl);

    $result{left} = \@itemsLeft;
    $result{right} = \@itemsRight;
    $result{center} = \@itemsCenter;
  }
  elsif ($location eq "sidemenu")
  {
    $itemsLeft[0] = Portal::Objects::MenuItem->new(type => "link", url => "$changePasswordUrl", langObj => $self->{langObj},
                    text => $changePasswordPhrase, title => $changePasswordPhrase, onMouseOver => "window.status='$changePasswordPhrase'; return true;");
    $itemsLeft[1] = Portal::Objects::MenuItem->new(type => "link", url => "$userInfoUrl", langObj => $self->{langObj},
                    text => "$userInfoPhrase", title => $userInfoPhrase, onMouseOver => "window.status='$userInfoPhrase'; return true;");
    $itemsLeft[2] = Portal::Objects::MenuItem->new(type => "link", url => "$editUserInfoUrl", langObj => $self->{langObj},
                    text => "$editUserInfoPhrase", title => $editUserInfoPhrase, onMouseOver => "window.status='$editUserInfoPhrase'; return true;");
    $itemsLeft[3] = Portal::Objects::MenuItem->new(type => "link", url => "$userPreferencesUrl", langObj => $self->{langObj},
                    text => "$editPhrase $userPreferencesPhrase", onMouseOver => "window.status='$editPhrase $userPreferencesPhrase'; return true;", title => "$editPhrase $userPreferencesPhrase");
    if ($state ne "Main")
    {
      $itemsLeft[4] = Portal::Objects::MenuItem->new(type => "seperator", langObj => $self->{langObj});
      $itemsLeft[5] = Portal::Objects::MenuItem->new(type => "link", url => "$mainMenuUrl", langObj => $self->{langObj},
                      text => $mainMenu, onMouseOver => "window.status='$mainMenu'; return true;", title => "$mainMenu");
    }

    $result{left} = \@itemsLeft;
  }

  return %result;
}

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

James A. Pattie (mailto:james@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::Base(3)

=cut
