# User.pm - The Object Class that provides a User Object
# Created by James A. Pattie, 11/07/2000.

# Copyright (c) 2000-2002, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::User;
use strict;
use Portal::AppState;
use Portal::Forms::User;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use Portal::Objects::UserPreferenceObject;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::AppState Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.06';

# global variables.
# modify to represent the commands this state provides.
# the name => value have the following meaning:
# name is the command this state recognizes.
# value is the langObj entry to use for the title when processing that command.
my %commands = ( display => "userInfo", editUser => "EditUserInfo", changePassword => "changePassword", preferences => "userPreferences", );

=head1 NAME

User - Object used to build a User Object Class.

=head1 SYNOPSIS

  use Portal::UserProperties::User;
  my $obj = Portal::UserProperties::User->new;
  if ($obj->error())
  {
    die $obj->errorMessage();
  }
  $obj->run(command => $command);

=head1 DESCRIPTION

User is a User class.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::User module.
 See Portal::AppState(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );

  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module
  $self->{commands} = \%commands;
  $self->{state} = "User";

  my $arguments = $self->arguments();
  eval "\$self->{userForms} = Portal::Forms::User->new($arguments);";
  if ($@)
  {
    $self->error($@);
    return $self;
  }
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return $self;
  }


  # do validation
  if (!$self->isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.
  $self->buildUrls;

  return $self;
}

=item % getCommands(void)

  This method returns the commands hash and is only to be
  used by the registerApp method for creating the permissions,
  if using the AppStateCommandSecurity implementation.

=cut
sub getCommands
{
  return %commands;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;

  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}

=item HTMLObject c_display()

=cut
sub c_display
{
  my $self = shift;
  my $doc = $self->prepDoc(template => "menubar+sidemenu-left");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }

  # call the Portal::Forms::User module to do the core display, validation and update work
  my %output = $self->{userForms}->viewUser(user => $self->{userObj}, companyObj => $self->{companyObj});
  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }

  # now display the output to the user.
  $doc->printTag(tag => "<maincontent>", value => $output{body});
  delete $output{body};
  # output the rest of the form info.
  $doc->print(%output);

  return $doc;
}

=item HTMLObject c_editUser

=cut
sub c_editUser
{
  my $self = shift;
  my $doc = $self->prepDoc(template => "menubar+sidemenu-left");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successEditPhrase = $self->langObj->map("successUserEdit");

  my $url = $self->url;

  # call the Portal::Forms::User module to do the core display, validation and update work
  my @result = $self->{userForms}->editUser(user => $self->{userObj},
      url => $url, app => $self->{app}, state => $self->{state});

  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }

  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->printTag(tag => "<maincontent>", value => $result[1]->{body});
    delete $result[1]->{body};
    # output the rest of the form info.
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[3]);
  }
  elsif ($result[0] == 1)
  {
    $doc->printTag(tag => "<maincontent>", value => $self->{methods}->displayMessageStr(doc => $doc, type => "status", message => $successEditPhrase,
                                     break => "below"));
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }

    # update the session for our userObj.
    $self->{portalSession}->{store}->{userObj} = $result[2];
    $self->{portalSession}->{store}->{changed} = 1;
  }

  return $doc;
}

=item HTMLObject c_changePassword

=cut
sub c_changePassword
{
  my $self = shift;
  my $doc = $self->prepDoc(template => "menubar+sidemenu-left");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successPasswordPhrase = $self->langObj->map("successPassword");

  my $url = $self->url;

  # call the Portal::Forms::User module to do the core display, validation and update work
  my @result = $self->{userForms}->changePassword(user => $self->{userObj},
      url => $url, app => $self->{app}, state => $self->{state});

  if ($self->{userForms}->error)
  {
    $self->error($self->{userForms}->errorMessage);
    return undef;
  }

  # now display the output to the user.
  if ($result[0] == 2)
  {
    $doc->printTag(tag => "<maincontent>", value => $result[1]->{body});
    delete $result[1]->{body};
    # output the rest of the form info.
    $doc->print(%{$result[1]});
    $doc->setOnload(code => $result[2]);
  }
  elsif ($result[0] == 1)
  {
    $doc->printTag(tag => "<maincontent>", value => $self->{methods}->displayMessageStr(doc => $doc, type => "status", message => $successPasswordPhrase,
                                     break => "below"));
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }

  return $doc;
}

=item HTMLObject c_preferences()

=cut
sub c_preferences
{
  my $self = shift;
  my $doc = $self->prepDoc(template => "menubar+sidemenu-left");
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  my $successPreferencesPhrase = $self->langObj->map("successPreferences");

  if (! exists $self->{input}->{update})
  {
    $doc = $self->displayUserPreferences(doc => $doc);
    if ($self->error)
    {
      $self->prefixError();
      return undef;
    }
  }
  else  # validate and update the users preferences.
  {
    my $colorScheme = $self->{input}->{colorScheme};
    my $dynamicContent = $self->{input}->{dynamicContent};
    my $frameBorder = $self->{input}->{frameBorder};
    my $frameScrolling = $self->{input}->{frameScrolling};
    my $frameWidth = $self->{input}->{frameWidth};
    my $frameHeight = $self->{input}->{frameHeight};

    # do validation
    my $errorString = "";
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my $systemColorScheme = grep($colorScheme, @systemColorSchemes);
    my $userColorScheme = grep($colorScheme, @userColorSchemes);
    my $invalidArgument = $self->{langObj}->map("invalidArgument");
    if ($systemColorScheme == 0 && $userColorScheme == 0)
    {
      $errorString .= sprintf($invalidArgument, "colorScheme", $colorScheme);
    }
    if ($dynamicContent !~ /^(1|0)$/)
    {
      $errorString .= sprintf($invalidArgument, "dynamicContent", $dynamicContent);
    }
    if ($frameBorder !~ /^(0|1)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameBorder", $frameBorder);
    }
    if ($frameScrolling !~ /^(yes|no|auto)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameScrolling", $frameScrolling);
    }
    if ($frameWidth !~ /^(\d+%?)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameWidth", $frameWidth);
    }
    if ($frameHeight !~ /^(\d+)$/)
    {
      $errorString .= sprintf($invalidArgument, "frameHeight", $frameHeight);
    }

    if (length $errorString > 0)
    {
      $doc = $self->displayUserPreferences(doc => $doc, error => $errorString);
      if ($self->error)
      {
        $self->prefixError();
        return undef;
      }
    }
    else
    {
      # update the user preferences.

      # handle colorScheme, dynamicContent
      foreach my $name ("colorScheme", "dynamicContent", "dynamicContentConfig")
      {
        my $configObjStr;
        if ($name eq "dynamicContentConfig")
        {
          # handle dynamicContentConfig
          my $configObj;
          eval { $configObj = Portal::XML::ConfigObject->new(langObj => $self->{langObj}); };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          $configObj->{version} = "1.0";
          $configObj->{module} = "DynamicContentConfig";
          $configObj->{settings}->{frameBorder} = $frameBorder;
          $configObj->{settings}->{frameScrolling} = $frameScrolling;
          $configObj->{settings}->{frameHeight} = $frameHeight;
          $configObj->{settings}->{frameWidth} = $frameWidth;

          eval { $configObjStr = $configObj->generateXML; };
          if ($@)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $@);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if (length $configObjStr == 0)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{langObj}->map("noConfigXMLGenerated"));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
        }

        my $module = ($name eq "colorScheme" ? "Global" : "Desktop");
        my $prefObj = Portal::Objects::UserPreferenceObject->new(id => $self->{userObj}->{id},
                      app => "Portal", module => $module, name => "$name",
                      value => ($name ne "dynamicContentConfig" ? $self->{input}->{$name} : $configObjStr), langObj => $self->{langObj});
        if ($prefObj->error)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $prefObj->errorMessage);
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        my $result = $self->{authObj}->updateUserPreference(userPreferenceObj => $prefObj);
        if ($self->{authObj}->error)
        {
          $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        if ($result == -1)
        { # have to create it
          $result = $self->{authObj}->createUserPreference(userPreferenceObj => $prefObj);
          if ($self->{authObj}->error)
          {
            $doc = $self->displayUserPreferences(doc => $doc, error => $self->{authObj}->errorMessage);
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          if ($result == -1)
          { # already existed!  don't know how!
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceNowExists"), $prefObj->print));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == -2)
          { # app doesn't exist
            $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->{langObj}->map("createUserPreferenceAppNotExist"), $prefObj->print));
            if ($self->error)
            {
              $self->prefixError();
              return undef;
            }
            return $doc;
          }
          elsif ($result == 1)
          { # we are ok.
          }
        }
        elsif ($result == -2)
        { # app doesn't exist
          $doc = $self->displayUserPreferences(doc => $doc, error => sprintf($self->langObj->map("updateUserPreferenceAppNotExist"), $prefObj->print));
          if ($self->error)
          {
            $self->prefixError();
            return undef;
          }
          return $doc;
        }
        elsif ($result == 1)
        { # we are ok.
        }
      }

      $doc = $self->displayUserPreferences(doc => $doc, status => $successPreferencesPhrase);
      if ($self->error)
      {
        $self->prefixError();
        return undef;
      }
    }
  }

  return $doc;
}

sub displayUserPreferences
{
  my $self = shift;
  my %args = ( doc => undef, error => "", status => "", @_ );
  my $doc = $args{doc};
  my $error = $args{error};
  my $status = $args{status};

  my $colorSchemePhrase = $self->langObj->map("colorScheme");
  my $dynamicContentPhrase = $self->langObj->map("dynamicContent");
  my $dynamicContentConfigPhrase = $self->langObj->map("dynamicContentConfig");
  my $frameBorderPhrase = $self->langObj->map("frameBorder");
  my $frameScrollablePhrase = $self->langObj->map("frameScrollable");
  my $frameWidthPhrase = $self->langObj->map("frameWidth");
  my $frameHeightPhrase = $self->langObj->map("frameHeight");
  my $editPhrase = $self->langObj->map("edit");
  my $userPreferencesPhrase = $self->langObj->map("userPreferences");

  # setup the XML parser needed.
  my $parserObj;
  eval { $parserObj = Portal::XML::ConfigParser->new(langObj => $self->{langObj}); };
  if ($@)
  {
    $self->error($@);
    return $doc;
  }

  # get the Defaults for the preferences.
  my %defaults = ();  # store the defaults the system gives us.
  foreach my $prefModule ("Global", "Desktop")
  {
    my @names = ($prefModule eq "Global" ? ("colorScheme") : ("dynamicContent", "dynamicContent_Config"));
    foreach my $prefName (@names)
    {
      my $prefNameDefault = "Portal_" . $prefModule . "_" . $prefName . "_Default";
      my $prefValueDefault = $self->{methods}->getConfigValue(name => $prefNameDefault, portalDB => $self->{applicationObj}->{portalDB});
      if ($self->{methods}->error)
      {
        $self->error($self->{methods}->errorMessage);
        return $doc;
      }
      if (!defined $prefValueDefault)
      {
        $self->error(sprintf($self->{langObj}->map("configEntryDoesNotExistUpgradeDatabase"), $prefNameDefault));
        return $doc;
      }
      if ($prefName eq "dynamicContent_Config" && $prefModule eq "Desktop")
      {
        # have to do the XML parsing
        my $configObj;
        eval { $configObj = $parserObj->parse(string => $prefValueDefault, module => "DynamicContentConfig", version => "1.0"); };
        if ($@)
        {
          $self->error(sprintf($self->{langObj}->map("parseFailed"), $@));
          return $doc;
        }
        foreach my $name (keys %{$configObj->{settings}})
        {
          $defaults{$prefModule}{$name} = $configObj->{settings}->{$name};
        }
      }
      else
      {
        $defaults{$prefModule}{$prefName} = $prefValueDefault;
      }
    }
  }

  # pull in the Portal preferences for the user.
  my @userPreferences = $self->{authObj}->getUserPreferences(userId => $self->{userObj}->{id},
                        app => "Portal");
  if ($self->{authObj}->error)
  {
    $self->error($self->{authObj}->errorMessage);
    return $doc;
  }
  my %preferences = ();  # store the preferences info that we get from the system.
  foreach my $prefObj (@userPreferences)
  {
    if ($prefObj->{name} eq "dynamicContentConfig" && $prefObj->{module} eq "Desktop")
    {
      # have to do the XML parsing
      my $configObj;
      eval { $configObj = $parserObj->parse(string => $prefObj->{value}, module => "DynamicContentConfig", version => "1.0"); };
      if ($@)
      {
        $self->error(sprintf($self->{langObj}->map("parseFailed"), $@));
        return $doc;
      }
      foreach my $name (keys %{$configObj->{settings}})
      {
        $preferences{$prefObj->{module}}{$name} = $configObj->{settings}->{$name};
      }
    }
    else
    {
      $preferences{$prefObj->{module}}{$prefObj->{name}} = $prefObj->{value};
    }
  }

  # verify we have preferences to edit, if not we specify the default.
  foreach my $module ("Global", "Desktop")
  {
    my @names = ($module eq "Global" ? ("colorScheme") : ("frameScrolling", "frameBorder", "frameWidth", "frameHeight", "dynamicContent"));
    foreach my $name (@names)
    {
      $preferences{$module}{$name} = (exists $self->{input}->{$name} ? $self->{input}->{$name} : (exists $preferences{$module}{$name} ? $preferences{$module}{$name} : $defaults{$module}{$name}));
    }
  }

  my $url = $self->url;
  my $output = "";

  if ($error)
  {
    $output .= $self->{methods}->displayMessageStr(doc => $doc, type => "error", message => $error,
                                     break => "below");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }
  if ($status)
  {
    $output .= $self->{methods}->displayMessageStr(doc => $doc, type => "status", message => $status,
                                     break => "below");
    if ($self->{methods}->error)
    {
      $self->error($self->{methods}->errorMessage);
      return undef;
    }
  }

  # generate the form to let them edit their Portal Preferences.
  $output .= <<"END_OF_CODE";
$editPhrase $userPreferencesPhrase
<br>
<form name="preferences" method="post" action="$url">
  <input type="hidden" name="app" value="$self->{app}">
  <input type="hidden" name="state" value="$self->{state}">
  <input type="hidden" name="command" value="preferences">
  <input type="hidden" name="update" value="1">
  <table border="0" cellpadding="2" cellspacing="0" class="edit">
    <tr>
      <td align="right">$colorSchemePhrase</td>
      <td align="left">
        <select name="colorScheme">
END_OF_CODE
    my @systemColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => "-1");
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    my @userColorSchemes = $self->{applicationObj}->getAvailableColorSchemes(userId => $self->{userObj}->{id});
    if ($self->{applicationObj}->error)
    {
      $self->error($self->{applicationObj}->errorMessage);
      return undef;
    }
    foreach my $option (@systemColorSchemes, @userColorSchemes)
    {
      my $selected = ($preferences{Global}{colorScheme} eq "$option" ? " selected" : "");
      $output .= "          <option value=\"$option\"$selected>$option\n";
    }
  $output .= <<"END_OF_CODE";
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$dynamicContentPhrase</td>
      <td align="left">
        <select name="dynamicContent">
END_OF_CODE
    foreach my $option ("1", "0")
    {
      my $selected = ($preferences{Desktop}{dynamicContent} eq "$option" ? " selected" : "");
      $output .= "          <option value=\"$option\"$selected>" . ($option ? "true" : "false") . "\n";
    }
  $output .= <<"END_OF_CODE";
        </select>
      </td>
    </tr>
    <tr>
      <td colspan="2" align="center"><b>$dynamicContentConfigPhrase</b></td>
    </tr>
    <tr>
      <td align="right">$frameBorderPhrase</td>
      <td align="left">
        <select name="frameBorder">
END_OF_CODE
    foreach my $option ("0", "1")
    {
      my $selected = ($preferences{Desktop}{frameBorder} eq "$option" ? " selected" : "");
      $output .= "          <option value=\"$option\"$selected>$option\n";
    }
  $output .= <<"END_OF_CODE";
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameScrollablePhrase</td>
      <td align="left">
        <select name="frameScrolling">
END_OF_CODE
    foreach my $option ("auto", "yes", "no")
    {
      my $selected = ($preferences{Desktop}{frameScrolling} eq "$option" ? " selected" : "");
      $output .= "          <option value=\"$option\"$selected>$option\n";
    }
  $output .= <<"END_OF_CODE";
        </select>
      </td>
    </tr>
    <tr>
      <td align="right">$frameWidthPhrase</td>
      <td align="left"><input type="text" name="frameWidth" value="$preferences{Desktop}{frameWidth}" size="4" maxlength="4"></td>
    </tr>
    <tr>
      <td align="right">$frameHeightPhrase</td>
      <td align="left"><input type="text" name="frameHeight" value="$preferences{Desktop}{frameHeight}" size="3" maxlength="3"></td>
    </tr>
  </table>
  <input type="submit" name="update" value="$editPhrase $userPreferencesPhrase">
</form>
    </td>
  </tr>
</table>
END_OF_CODE

  $doc->printTag(tag => "<maincontent>", value => $output);

  return $doc;
}

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::UserProperties(3), Portal::AppState(3)

=cut
