# SOAP.pm - The Class that provides SOAP methods
# Created by James A. Pattie, 2004-07-05.

# Copyright (c) 2004, Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

package Portal::UserProperties::SOAP;
use strict;
use Portal::SOAP;
use Portal::Objects::SOAPResult;
use vars qw($AUTOLOAD $VERSION @ISA @EXPORT);

require Exporter;

@ISA = qw(Portal::SOAP Exporter AutoLoader);
@EXPORT = qw();

$VERSION = '0.01';

# global variables.

=head1 NAME

SOAP - Object used to provide SOAP methods for Portal::UserProperties.

=head1 SYNOPSIS

  use Portal::UserProperties::SOAP;
  my $obj = Portal::UserProperties::SOAP->new(langObj => $langObj);
  if ($obj->error())
  {
    die $obj->errorMessage();
  }

=head1 DESCRIPTION

SOAP is a class of SOAP methods.

=head1 Exported FUNCTIONS

B<NOTE>: I<bool> = 1(true), 0(false)

=over 4

=item scalar new()

 Creates a new instance of the Portal::UserProperties::SOAP module.
 See Portal::SOAP(3) for a listing of required arguments.

=cut

sub new
{
  my $class = shift;
  my $self = $class->SUPER::new(@_);
  my %args = ( @_ );


  if ($self->error)
  {
    $self->prefixError();
    return $self;
  }

  # instantiate anything unique to this module

  # do validation, allowing ourselves to be derived from.
  if (!$self->Portal::UserProperties::SOAP::isValid)
  {
    # the error is set in the isValid() method.
    return $self;
  }

  # do anything else you might need to do.

  return $self;
}

=item bool isValid(void)

 Returns 0 or 1 to indicate if the object is valid.
 The error will be available via errorMessage().

=cut

sub isValid
{
  my $self = shift;


  # make sure our Parent class is valid.
  if (!$self->SUPER::isValid())
  {
    $self->prefixError();
    return 0;
  }

  # validate our parameters.

  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    return 0;
  }

  return 1;
}


=item SOAPResult cleanupCompany(companyObj, appObj, companyAppObj, dataDir)

 takes: dataDir
 returns: SOAPResult object or undef on major error
 This routine dumps any data necessary and if a database is defined, 
 destroys the database.
 
=cut
sub cleanupCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, appObj => undef, companyAppObj => undef, dataDir => "", @_ );
  my $companyObj = $args{companyObj};
  my $appObj = $args{appObj};
  my $companyAppObj = $args{companyAppObj};
  my $dataDir = $args{dataDir};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (length $dataDir == 0 || ! -d $dataDir)
  {
    $self->invalid("dataDir", $dataDir, " - is invalid or does not exist!");
  }
  
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  
  # fixup the portalDB objects for the necessary objects.  
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  
  # make sure the caller is a sysadmin or an admin in the company being removed.
  if (!$self->{caller}->{admin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be an Admin or a Sysadmin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not cleanup someone elses company!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }

  # don't forget to fixup the dataDir value to have your appName, etc.
  $dataDir .= "/" . $appObj->{name} . "/";

  # do any cleanup necessary
  
  $self->cleanupDB();
  return $result;
}

=item SOAPResult setupCompany(companyObj, appObj, companyAppObj)

 returns: SOAPResult object or undef on major error
 This routine proceeds to create a database for the company if 
 needed and does any initialization necessary.  
 
=cut
sub setupCompany
{
  my $self = shift;
  my %args = ( companyObj => undef, appObj => undef, companyAppObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $appObj = $args{appObj};
  my $companyAppObj = $args{companyAppObj};

  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }
  
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  
  # fixup the portalDB objects for the necessary objects.  
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  
  # make sure the caller is a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }
  # do any setup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult setupUser(companyObj, userObj, appObj, companyAppObj)

 returns: SOAPResult object or undef on major error
 This routine configures the user in the application.  
 
=cut
sub setupUser
{
  my $self = shift;
  my %args = ( companyObj => undef, userObj => undef, appObj => undef, companyAppObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};
  my $appObj = $args{appObj};
  my $companyAppObj = $args{companyAppObj};
  
  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }
  
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  
  # fixup the portalDB objects for the necessary objects.  
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  
  # make sure the caller is a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }
  
  # do any setup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult updateUser(companyObj, userObj, appObj, companyAppObj)

 returns: SOAPResult object or undef on major error
 This routine updates the users settings in the application.  
 This is mainly to synchronize any changes that were made to 
 the users info in the portal_db that the application duplicates 
 or relies on for user settings in its own database.  
 
=cut
sub updateUser
{
  my $self = shift;
  my %args = ( companyObj => undef, userObj => undef, appObj => undef, companyAppObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};
  my $appObj = $args{appObj};
  my $companyAppObj => $args{companyAppObj};
  
  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }
  
  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  
  # fixup the portalDB objects for the necessary objects.  
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  
  # make sure the caller is either the user, a sysadmin or an admin in the company being setup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin} && $self->{caller}->{uname} != $userObj->{uname})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin, Admin or the user being updated!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not setup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }
  
  # do any setup necessary

  $self->cleanupDB();
  return $result;
}

=item SOAPResult cleanupUser(companyObj, userObj, appObj, companyAppObj)
 
 returns: SOAPResult object or undef on major error
 This routine archives any info on the user and then removes 
 their data from active use.
 
=cut
sub cleanupUser
{
  my $self = shift;
  my %args = (companyObj => undef, userObj => undef, appObj => undef, companyAppObj => undef, @_ );
  my $companyObj = $args{companyObj};
  my $userObj = $args{userObj};
  my $appObj = $args{appObj};
  my $companyAppObj = $args{companyAppObj};
  
  # start our result object, default to a successfull result code.
  my $result = Portal::Objects::SOAPResult->new(langObj => $self->{langObj}, result => 1);
  if ($result->error)
  {
    $self->error($result->errorMessage);
    return undef;
  }

  if (!defined $companyObj)
  {
    $self->missing("companyObj");
  }
  elsif (!$companyObj->isValid)
  {
    $self->invalid("companyObj", $companyObj->errorMessage);
  }
  if (!defined $userObj)
  {
    $self->missing("userObj");
  }
  elsif (!$userObj->isValid)
  {
    $self->invalid("userObj", $userObj->errorMessage);
  }
  if (!defined $appObj)
  {
    $self->missing("appObj");
  }
  elsif (!$appObj->isValid)
  {
    $self->invalid("appObj", $appObj->errorMessage);
  }
  if (!defined $companyAppObj)
  {
    $self->missing("companyAppObj");
  }
  elsif (!$companyAppObj->isValid)
  {
    $self->invalid("companyAppObj", $companyAppObj->errorMessage);
  }
  
  # fixup the portalDB objects for the necessary objects.  
  $self->setupDB();
  if ($self->error)
  {
    $self->prefixError();
    return undef;
  }
  
  # make sure the caller is a sysadmin or an admin in the company being cleanedup.
  if (!$self->{caller}->{admin} && !$self->{caller}->{sysadmin})
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You must be a Sysadmin or Admin!");
  }
  if (!$self->{caller}->{sysadmin} && ($self->{caller}->{admin} && $self->{caller}->{companyId} != $companyObj->{id}))
  {
    $self->invalid("caller", $self->{caller}->{uname}, "You can not cleanup someone elses company!");
  }
  # make sure the userObj is part of the companyObj.
  if ($userObj->{companyId} != $companyObj->{id})
  {
    $self->invalid("user", $userObj->{uname}, "Is not in the same company as $companyObj->{name}!");
  }
  if ($self->numInvalid() > 0 || $self->numMissing() > 0)
  {
    $self->error($self->genErrorString("all"));
    $self->cleanupDB();
    return undef;
  }
  
  # do any cleanup necessary
  
  $self->cleanupDB();
  return $result;
}

=back

=cut

1;
__END__

=head1 NOTE

 All data fields are accessible by specifying the object
 and pointing to the data member to be modified on the
 left-hand side of the assignment.
 Ex.  $obj->variable($newValue); or $value = $obj->variable;

=head1 AUTHOR

Xperience, Inc. (mailto:admin@pcxperience.com)

=head1 SEE ALSO

perl(1), Portal(3), Portal::SOAP(3), Portal::Objects::SOAPResult(3)

=cut

