#! /usr/bin/perl
# setup.pl - This script is run to configure the App in the Portal.
# Created by James Pattie, 07/02/2001.  Copyright (c) 2000-2002 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

use strict;
use Portal::Language;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;

my $debug = 0;  # display any of the debug info (off by default).

my $configVersion = "0.2";  # Current Configuration File Version to be working with.
my $appName = "Example";   # Change these values to reflect the "App" being configured.
my $dbName = "portal_db";  # specify the name of your database if you need one.
my $dbType = "Pg";  # either Pg for PostgreSQL or mysql for MySQL.
my $dbPerCompany = 0;  # 1 if you need a database per company.
my $appType = "tools";  # one of administration, help, bugs, regular, tools or news
my $appAdminType = "normal";  # If appType = administration, then you can have a 'normal' or 'system' admin app.
my $appDescription = "Hello World App";
my $appIconName = "example.jpg";
my $helpDefined = 1;  # specify 0 if no Help.pm module created.

# default values to use.
my $isAppFree = 1;  # Set to 0 if app is not free.
my $appCost = "0.00";  # Set to price if app is not free.
my $unit = 0;  # 0 = per month, 1 = per user.
my $numLicenses = 0;  # 0 means unlimited.  Can only be changed if app is not free.  Is only used if assigning to all companies.

my $wapAware = 0;  # Is this app Wap Aware?

print "\n#################  Portal::$appName Configuration settings #################\n";

my $configFile = "/etc/pcx_portal/$appName.xml";

# make sure the config file exists.
if (! -e $configFile)
{
  die "Error:  Config file = '$configFile' does not exist!\n";
}

my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->didErrorOccur)
{
  die($langObj->errorMessage);
}

my $configObj = Portal::XML::ConfigParser->new(langObj => $langObj);
if ($configObj->didErrorOccur)
{
  die $configObj->errorMessage;
}
my $settings = undef;
my $done = 0;
my $version = $configVersion;

# pull in the settings from the config file.
while (!$done)
{
  eval { $settings = $configObj->parse(file => $configFile, module => "$appName", version => $version); };
  if ($@)
  {
    if ($@ =~ /(ConfigParser->validateVersion\(\) $configObj->{errorPhrase}  '(\d+\.\d+)' is not equal to Version '$version'!)/m)
    {
      $version = $2;  # set the version to what the config file is.
      print "Downgrading to version '$version'\n" if ($debug);
    }
    else
    {
      die $@;
    }
  }
  else
  {
    $done = 1;  # we are done.
  }
}

RESTART:  # We go here if they say No to use current settings.
my $useCurrent = 0;
my $input = "";
my $configItems = Portal::XML::ConfigObject->new(langObj => $langObj);

# Make sure we get all values that were there before so that they get put back out if we don't modify them.
$configItems->version($settings->version());
print "version = '$configItems->{settings}->{version}'\n" if ($debug);
$configItems->module($settings->module());
print "module = '$configItems->{settings}->{module}'\n" if ($debug);
foreach my $setting (keys %{$settings->settings()})
{
  # fixup any of the old boolean true/false entries -> 1/0.
  if ($setting =~ /^(autorun|assignToAdminUsers|assignToAllCompanies|assignToAllUsers|assignToSysAdminUsers|isAppFree)$/)
  {
    if ($settings->settings->{$setting} eq "true")
    {
      $settings->settings->{$setting} = 1;
    }
    if ($settings->settings->{$setting} eq "false")
    {
      $settings->settings->{$setting} = 0;
    }
  }
  $configItems->settings->{$setting} = $settings->settings->{$setting};
  print "$setting = '$configItems->{settings}->{$setting}'\n" if ($debug);
}

if ($configItems->{settings}->{configVersion} eq "0.1")
{
  # This is where you could put in anything that is new that you won't be asking
  # the user for the value of (color constants, etc.)
  $configItems->{settings}->{configVersion} = "0.2";
}

$configItems->{settings}->{appType} = $appType;
$configItems->{settings}->{appAdminType} = $appAdminType;
$configItems->{settings}->{appDescription} = $appDescription;
$configItems->{settings}->{appIconName} = $appIconName;
$configItems->{settings}->{dbType} = $dbType;
$configItems->{settings}->{dbPerCompany} = $dbPerCompany;
$configItems->{settings}->{helpDefined} = $helpDefined;

# ****************************************************

print "\nDatabase Type = '$configItems->{settings}->{dbType}'\n";

# get the Database Port
$configItems->{settings}->{dbPort} = getInput(prompt => "Enter Database Server Port", value => ($settings->{settings}->{dbPort} ne "DBPORT" ? $settings->{settings}->{dbPort} : ($configItems->{settings}->{dbType} eq "Pg" ? "5432" : "3306")), mask => "\\d+");

print "\nDatabase Per Company = $dbPerCompany\n";
# get the Database Name
if ($configItems->{settings}->{dbPerCompany})
{
  $configItems->{settings}->{dbName} = getInput(prompt => "Enter Database Name to append to the Company Name", value => ($settings->{settings}->{dbName} ne "DBNAME" ? $settings->{settings}->{dbName} : "$dbName"));
}
else
{
  $configItems->{settings}->{dbName} = $dbName;
}

$configItems->{settings}->{width} = getInput(prompt => "Default Width of App Window", value => ($settings->{settings}->{width} ne "WIDTH" ? $settings->{settings}->{width} : "640"), mask => "\\d+");
$configItems->{settings}->{height} = getInput(prompt => "Default Height of App Window", value => ($settings->{settings}->{height} ne "HEIGHT" ? $settings->{settings}->{height} : "480"), mask => "\\d+");
$configItems->{settings}->{autorun} = getInput(prompt => "Should App default to automatically start when a user logs in?  (1 or 0)", value => ($settings->{settings}->{autorun} ne "AUTORUN" ? $settings->{settings}->{autorun} : 0), mask => "(1|0)");

print "\nAppType = '$appType'\n";
print "Administration Type = '$appAdminType'\n" if ($appType eq "administration");

$configItems->{settings}->{isAppFree} = getInput(prompt => "Is this App Free? (1 or 0)", value => ($settings->{settings}->{isAppFree} ne "ISAPPFREE" ? $settings->{settings}->{isAppFree} : $isAppFree), mask => "(1|0)");
if ($configItems->{settings}->{isAppFree} == 0)
{
  $configItems->{settings}->{appCost} = getInput(prompt => "Cost of App", value => ($settings->{settings}->{appCost} ne "APPCOST" ? $settings->{settings}->{appCost} : $appCost), mask => "\d+\.\d{2}");
  $configItems->{settings}->{unit} = getInput(prompt => "Per Month (0) or Per User (1)", value => ($settings->{settings}->{unit} ne "UNIT" ? $settings->{settings}->{unit} : $unit), mask => "(0|1)");
  $configItems->{settings}->{numLicenses} = getInput(prompt => "Number of licenses to defaultly assign", value => ($settings->{settings}->{numLicenses} ne "NUMLICENSES" ? $settings->{settings}->{numLicenses} : $numLicenses), mask => "\d+");
}
else
{
  $configItems->{settings}->{appCost} = "0.00";
  $configItems->{settings}->{unit} = 0;
  $configItems->{settings}->{numLicenses} = 0;
}

$configItems->{settings}->{assignToAllCompanies} = getInput(prompt => "Assign to All Companies (1 or 0) (If 0, then no assignment to users will be done)", value => ($settings->{settings}->{assignToAllCompanies} ne "ASSIGNTOALLCOMPANIES" ? $settings->{settings}->{assignToAllCompanies} : 1), mask => "(1|0)");
if ($configItems->{settings}->{assignToAllCompanies})
{
  if ($appType eq "administration")
  {
    $configItems->{settings}->{assignToAllUsers} = 0;  # This isn't possible with an administration app.
    if ($appAdminType eq "normal")
    {
      $configItems->{settings}->{assignToAdminUsers} = getInput(prompt => "Assign to All Admin Users (1 or 0)", value => ($settings->{settings}->{assignToAdminUsers} ne "ASSIGNTOADMINUSERS" ? $settings->{settings}->{assignToAdminUsers} : 1), mask => "(1|0)");
      if ($configItems->{settings}->{assignToAdminUsers} == 0)
      {
        $configItems->{settings}->{assignToSysAdminUsers} = getInput(prompt => "Assign Only to System Admin Users (1 or 0)", value => ($settings->{settings}->{assignToSysAdminUsers} ne "ASSIGNTOSYSADMINUSERS" ? $settings->{settings}->{assignToSysAdminUsers} : 1), mask => "(1|0)");
      }
      else
      {
        $configItems->{settings}->{assignToSysAdminUsers} = 0;  # don't need this set as the AdminUsers also covers SysAdmin users.
      }
    }
    else
    {
      $configItems->{settings}->{assignToAdminUsers} = 0;  # This can't be done with a SysAdmin App.
      $configItems->{settings}->{assignToSysAdminUsers} = getInput(prompt => "Assign to System Admin Users (1 or 0)", value => ($settings->{settings}->{assignToSysAdminUsers} ne "ASSIGNTOSYSADMINUSERS" ? $settings->{settings}->{assignToSysAdminUsers} : 1), mask => "(1|0)");
    }
  }
  else
  {
    $configItems->{settings}->{assignToAllUsers} = getInput(prompt => "Assign To All Users (1 or 0)", value => ($settings->{settings}->{assignToAllUsers} ne "ASSIGNTOALLUSERS" ? $settings->{settings}->{assignToAllUsers} : 1), mask => "(1|0)");
    if ($configItems->{settings}->{assignToAllUsers} == 0)
    {
      $configItems->{settings}->{assignToAdminUsers} = getInput(prompt => "Assign to All Admin Users (1 or 0)", value => ($settings->{settings}->{assignToAdminUsers} ne "ASSIGNTOADMINUSERS" ? $settings->{settings}->{assignToAdminUsers} : 1), mask => "(1|0)");
      if ($configItems->{settings}->{assignToAdminUsers} == 0)
      {
        $configItems->{settings}->{assignToSysAdminUsers} = getInput(prompt => "Assign Only to System Admin Users (1 or 0)", value => ($settings->{settings}->{assignToSysAdminUsers} ne "ASSIGNTOSYSADMINUSERS" ? $settings->{settings}->{assignToSysAdminUsers} : 1), mask => "(1|0)");
      }
      else
      {
        $configItems->{settings}->{assignToSysAdminUsers} = 0;  # don't need this set as the AdminUsers also covers SysAdmin users.
      }
    }
    else
    { # no need to set the Admin or SysAdmin flags as they are automatically included.
      $configItems->{settings}->{assignToAdminUsers} = 0;
      $configItems->{settings}->{assignToSysAdminUsers} = 0;
    }
  }
}
else
{
  $configItems->{settings}->{assignToAllUsers} = 0;
  $configItems->{settings}->{assignToAdminUsers} = 0;
  $configItems->{settings}->{assignToSysAdminUsers} = 0;
}

$configItems->{settings}->{wapAware} = $wapAware;

# ****************************************************

# verify they want to use these settings.
$input = getInput(prompt => "Accept these values (Y) or Redo (N)?", value => "Y", mask => "Y|N|y|n");
if ($input =~ /^N$/i)
{
  goto RESTART;
}

# validate the configuration object.
my @result = $configItems->isValid;
if (!$result[0])
{
  die "Portal settings not valid!\n\n" . join("\n", @{$result[1]}) . "\n";
}

print "Generating Config file '$configFile'...\n";
print $configItems->generateXML if ($debug);  # debug output.

# generate the Config file now.
open (F, ">$configFile") or die "Error:  Couldn't open '$configFile'!  Error = $!\n";

print F $configItems->generateXML;

close (F)		or die "can't close $configFile: $!";

print "\nDon't forget to run appinstall.pl to actually register this app with the Portal.\n";

# getInput - prompt, mask, and value
sub getInput
{
  my %args = (prompt => "", mask => ".*", value => "", @_);
  my $prompt = $args{prompt};
  my $mask = $args{mask};
  my $value = $args{value};

  my $input = "";
  my $done = 0;
  while (!$done)
  {
    print "\n$prompt [$value]: ";
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $value;
    }
    if ($input !~ /^($mask)$/)
    {
      print "'$input' is invalid!\n";
      $input = "";
    }
    else
    {
      $done = 1;  # This will let me out even if I'm entering an empty string, as long as that is valid.
    }
  }
  $input =~ s/@/\\@/g;  # make sure that any @'s are escaped.
  $input =~ s/%/\\%/g;  # make sure that any %'s are escaped.
  return $input;
}

1;
