#! /usr/bin/perl
# This script will install the application into the Portal and do any default assignments to companies and users as defined
# by the configuration settings in the Application.

# appinstall.pl - Created by James Pattie, (james@pcxperience.com)
# Copyright (c) 2000-2003, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 11/24/2000

use Getopt::Long;
use Portal::Language;
use Portal::Data::Config;
use Portal::Auth;
use Portal::Log;
use Portal::Application;
use Portal::Objects::ApplicationObject;
use Portal::Objects::AppServerObject;
use Portal::Objects::CompanyApplicationObject;
use Portal::Objects::UserApplicationObject;
use Portal::Data::Variables;
use Portal::Methods;
use strict;

$| = 1;

my $syntax = "appinstall.pl --appname appName\n";

# get the command line info so we know what application we are trying to assign to the Portal.
my $appName = "";
GetOptions('appname=s' => \$appName);

if (length $appName == 0)
{
  die "Error:  You did not specify the appname to use!\n$syntax";
}

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = undef;
eval "\$configObj = Portal::Data::Config->new(langObj => \$langObj);";
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($configObj->error)
{
  die $configObj->errorMessage;
}
my $appConfigObj = undef;
eval "\$appConfigObj = Portal::Data::Config->new(langObj => \$langObj, app => \$appName);";
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($appConfigObj->error)
{
  die $appConfigObj->errorMessage;
}

my $methods = Portal::Methods->new(langObj => $langObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}

# do quick validation of values in the $appName Portal::Data::Config object that we rely upon for this installation to succeed.
my $portalVariables = Portal::Data::Variables->new(langObj => $langObj);
if (length $appConfigObj->get("dbName") == 0)
{
  myDie(error => "dbName = '" . $appConfigObj->get("dbName") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("dbPerCompany") !~ /^(1|0)$/)
{
  myDie(error => "dbPerCompany = '" . $appConfigObj->get("dbPerCompany") . "' is invalid!\n", configObj => $configObj);
}
if (!exists $portalVariables->{appTypes}->{$appConfigObj->get("appType")})
{
  myDie(error => "appType = '" . $appConfigObj->get("appType") . "' is invalid!\n", configObj => $configObj);
}
if (!exists $portalVariables->{adminTypes}->{$appConfigObj->get("appAdminType")})
{
  myDie(error => "appAdminType = '" . $appConfigObj->get("appAdminType") . "' is invalid!\n", configObj => $configObj);
}
if (length $appConfigObj->get("appDescription") == 0)
{
  myDie(error => "appDescription = '" . $appConfigObj->get("appDescription") . "' is invalid!\n", configObj => $configObj);
}
if (length $appConfigObj->get("appIconName") == 0)
{
  myDie(error => "appIconName = '" . $appConfigObj->get("appIconName") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("height") < 20)
{
  myDie(error => "height = '" . $appConfigObj->get("height") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("width") < 20)
{
  myDie(error => "width = '" . $appConfigObj->get("width") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("autorun") !~ /^(1|0)$/)
{
  myDie(error => "autorun = '" . $appConfigObj->get("autorun") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("dbType") !~ /^(Pg|mysql)$/)
{
  myDie(error => "dbType = '" . $appConfigObj->get("dbType") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("dbPort") !~ /^\d+$/)
{
  myDie(error => "dbPort = '" . $appConfigObj->get("dbPort") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("isAppFree") !~ /^(1|0)$/)
{
  myDie(error => "isAppFree = '" . $appConfigObj->get("isAppFree") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("assignToAllCompanies") !~ /^(1|0)$/)
{
  myDie(error => "assignToAllCompanies = '" . $appConfigObj->get("assignToAllCompanies") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("assignToAllUsers") !~ /^(1|0)$/)
{
  myDie(error => "assignToAllUsers = '" . $appConfigObj->get("assignToAllUsers") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("assignToAdminUsers") !~ /^(1|0)$/)
{
  myDie(error => "assignToAdminUsers = '" . $appConfigObj->get("assignToAdminUsers") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("assignToSysAdminUsers") !~ /^(1|0)$/)
{
  myDie(error => "assignToSysAdminUsers = '" . $appConfigObj->get("assignToSysAdminUsers") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("appCost") !~ /^\d+\.\d{2}$/)
{
  myDie(error => "appCost = '" . $appConfigObj->get("appCost") . "' is invalid!\n", configObj => $configObj);
}
if (!exists $portalVariables->{units}->{$appConfigObj->get("unit")})
{
  myDie(error => "unit = '" . $appConfigObj->get("unit") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("numLicenses") < 0)
{
  myDie(error => "numLicenses = '" . $appConfigObj->get("numLicenses") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("wapAware") !~ /^(1|0)$/)
{
  myDie(error => "wapAware = '" . $appConfigObj->get("wapAware") . "' is invalid!\n", configObj => $configObj);
}

if ($appConfigObj->get("prepDocTemplate") !~ /^(none|title|menubar(\+sidemenu\-(left|right|detect)|\-vertical)?|sidemenu\-(left|right)|custom)$/)
{
  myDie(error => "prepDocTemplate = '" . $appConfigObj->get("prepDocTemplate") . "' is invalid!\n", configObj => $configObj);
}
if ($appConfigObj->get("prepDocTemplate") eq "custom" && length $appConfigObj->get("prepDocCustomTemplate") == 0)
{
  myDie(error => "prepDocCustomTemplate = '" . $appConfigObj->get("prepDocCustomTemplate") . "' is invalid!\n", configObj => $configObj);
}

# do some quick validations of assignment, cost, etc.
if ($appConfigObj->get("isAppFree") && $appConfigObj->get("appCost") ne "0.00")
{
  myDie(error => "You cannot specify a cost other than '0.00' for a Free app!\n", configObj => $configObj);
}
if ($appConfigObj->get("isAppFree") && $appConfigObj->get("numLicenses") != 0)
{
  myDie(error => "You cannot specify a number of Licenses other than '0' for a Free app!\n", configObj => $configObj);
}

if ($appConfigObj->get("appType") eq "administration" && $appConfigObj->get("appAdminType") eq "system" && $appConfigObj->get("assignToAllUsers"))
{
  myDie(error => "You cannot assign a Systems Administration app to all users!\n", configObj => $configObj);
}

if ($appConfigObj->get("assignToAllCompanies") == 0 && $appConfigObj->get("assignToAllUsers"))
{
  myDie(error => "You cannot assign this app to all users if you do not assign it first to all companies!\n", configObj => $configObj);
}

# make connection to the portal database

my $portalDB = $methods->portalDBSetup(type => "portal", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($portalDB->error)
{
  myDie(error => $portalDB->errorMessage, configObj => $configObj);
}

my $billingDB = $methods->portalDBSetup(type => "billing", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($billingDB->error)
{
  myDie(error => $billingDB->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Application module.
my $applicationObj = Portal::Application->new(portalDB => $portalDB, langObj => $langObj);
if ($applicationObj->error)
{
  myDie(error => $applicationObj->errorMessage, configObj => $configObj);
}

# define the application from the config data.
my $appObj = Portal::Objects::ApplicationObject->new(id => 0, name => $appName, description => $appConfigObj->get("appDescription"),
             iconName => $appConfigObj->get("appIconName"), server => $configObj->get("myHostName"), port => ($configObj->get("httpType") eq "http" ? 80 : 443),
             cost => $appConfigObj->get("appCost"), unit => $appConfigObj->get("unit"), type => $appConfigObj->get("appType"), adminType => $appConfigObj->get("appAdminType"),
             dbName => $appConfigObj->get("dbName"), height => $appConfigObj->get("height"), width => $appConfigObj->get("width"), autorun => $appConfigObj->get("autorun"),
             number => $appConfigObj->get("numLicenses"), dbHost => $configObj->get("dbHost"), dbType => $appConfigObj->get("dbType"), dbPort => $appConfigObj->get("dbPort"), wap => $appConfigObj->get("wapAware"), langObj => $langObj);
if ($appObj->error)
{
  myDie(error => $appObj->errorMessage, configObj => $configObj);
}

# see if we are already installed.
my $tmpAppObj = $applicationObj->getApplicationInfo(name => $appObj->get("name"));
if (!defined $tmpAppObj)
{
  if ($applicationObj->error)
  {
    myDie(error => $applicationObj->errorMessage, configObj => $configObj);
  }
  else  # not created yet!
  {
    my $result = $applicationObj->createApplication(applicationObj => $appObj);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if ($result == 1)
    {
      # get the Id of the newly created application.
      $tmpAppObj = $applicationObj->getApplicationInfo(name => $appObj->get("name"));
      if (!defined $tmpAppObj)
      {
        if ($applicationObj->error)
        {
          myDie(error => $applicationObj->errorMessage, configObj => $configObj);
        }
        else
        {
          myDie(error => "Just created the application but it doesn't exist when getting it's ID!", configObj => $configObj);
        }
      }
      else
      {
        if ($tmpAppObj->error)
        {
          myDie(error => "Getting ID of newly created Application.  " . $tmpAppObj->errorMessage, configObj => $configObj);
        }
        print "Created Application (" . $appObj->get("name") . "), ID = '" . $tmpAppObj->get("id") . "'.\n";
      }
    }
    elsif ($result == -1)
    {
      myDie(error => "The application just said it already exists but we got here because we couldn't find it before!", configObj => $configObj);
    }
  }
}
else
{
  if ($tmpAppObj->error)
  {
    myDie(error => $tmpAppObj->errorMessage, configObj => $configObj);
  }
  else  # already exist.  See if we need to make a backup entry first.
  {
    if ($tmpAppObj->get("server") ne $appObj->get("server") || $tmpAppObj->get("port") ne $appObj->get("port"))  # servers and/or port are different
    {
      my $appServerObj = Portal::Objects::AppServerObject->new(id => $tmpAppObj->get("id"), server => $tmpAppObj->get("server"), port => $tmpAppObj->get("port"),
                         dbHost => $tmpAppObj->get("dbHost"), dbType => $tmpAppObj->get("dbType"), dbPort => $tmpAppObj->get("dbPort"), langObj => $langObj);
      if ($appServerObj->error)
      {
        myDie(error => $appServerObj->errorMessage, configObj => $configObj);
      }
      # create the server backup entry
      my $result = $applicationObj->backupAppServerEntry(appServerObj => $appServerObj);
      if ($applicationObj->error)
      {
        myDie(error => $applicationObj->errorMessage, configObj => $configObj);
      }
      if ($result == 1) # ok
      {
        print "Backed up Application Server Entry.  Server = '". $appServerObj->get("server") . "', Port = '" . $appServerObj->get("port") . "', DBHost = '" . $appServerObj->get("dbHost") . "', DBType = '" . $appServerObj->get("dbType") . "', DBPort = '" . $appServerObj->get("dbPort") . "'.\n";
      }
      elsif ($result == -1) # already done
      {
        print "Application Server Entry is already backed up.  Server = '" . $appServerObj->get("server") . "', Port = '" . $appServerObj->get("port") . "', DBHost = '" . $appServerObj->get("dbHost") . "', DBType = '" . $appServerObj->get("dbType") . "', DBPort = '" . $appServerObj->get("dbPort") . "'.\n";
      }
      elsif ($result == -2) # app doesn't exist
      {
        myDie(error => "When trying to backup the Application Server entry, the routine said the application doesn't exist!", configObj => $configObj);
      }

      $appObj->set("id" => $tmpAppObj->get("id"));

      # now update the server entry to have our settings.
      $result = $applicationObj->updateApplication(applicationObj => $appObj);
      if ($applicationObj->error)
      {
        myDie(error => $applicationObj->errorMessage, configObj => $configObj);
      }
      if ($result == 1) # ok
      {
        print "Updated Application Settings ok.\n";
      }
      elsif ($result == -1) # app doesn't exist
      {
        myDie(error => "When trying to update the Application entry, the routine said the application doesn't exist!", configObj => $configObj);
      }
      $tmpAppObj = $appObj;  # make sure we are using the right info.
    }
    else  # same installation.
    {
      print "The Application is already installed and doesn't need to be backed up. -- Updating anyway!\n";

      $appObj->set("id" => $tmpAppObj->get("id"));

      # now update the server entry to have our settings.
      my $result = $applicationObj->updateApplication(applicationObj => $appObj);
      if ($applicationObj->error)
      {
        myDie(error => $applicationObj->errorMessage, configObj => $configObj);
      }
      if ($result == 1) # ok
      {
        print "Updated Application Settings ok.\n";
      }
      elsif ($result == -1) # app doesn't exist
      {
        myDie(error => "When trying to update the Application entry, the routine said the application doesn't exist!", configObj => $configObj);
      }
      $tmpAppObj = $appObj;  # make sure we are using the right info.
    }
  }
}

$appObj = $tmpAppObj;  # make sure we have the correct info in appObj.

# instantiate an instance of the Auth module.
my $authObj = Portal::Auth->new(portalDB => $portalDB, billingDB => $billingDB, langObj => $langObj);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

# see if we need to assign this application to all companies
if ($appConfigObj->get("assignToAllCompanies"))
{
  # get all companies in the Portal.
  my @Companies = $authObj->getListOfCompanies;
  if ($authObj->error)
  {
    myDie(error => $authObj->errorMessage, configObj => $configObj);
  }

  # now iterate over all companies and make the app assignment to them.
  foreach my $companyObj (@Companies)
  {
    my $dbHost = $appObj->get("dbHost");

    # see if the assignment exists.
    my $tmpCompanyAppObj = $applicationObj->getCompanyAppInfo(companyId => $companyObj->get("id"), appId => $appObj->get("id"));
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if (defined $tmpCompanyAppObj)
    {
      $dbHost = $tmpCompanyAppObj->get("dbHost");
    }
    else
    {
      # ask the user what host their database is going to be on.
      $dbHost = getInput(prompt => "What is the IP/host name of your database server?", mask => "/^(.+)\$/", value => $dbHost);
    }

    # define the CompanyApplicationObject for this company/app pair.
    my $dbName = ($appConfigObj->get("dbPerCompany") ? $companyObj->get("code") . "_" : "") . $appConfigObj->get("dbName");
    my $companyAppObj = Portal::Objects::CompanyApplicationObject->new(langObj => $langObj);
    if ($companyAppObj->error)
    {
      myDie(error => $companyAppObj->errorMessage, configObj => $configObj);
    }
    $companyAppObj->populate(appId => $appObj->get("id"), companyId => $companyObj->get("id"), number => $appConfigObj->get("numLicenses"),
                        server => $appObj->get("server"), port => $appObj->get("port"), cost => $appObj->get("cost"), unit => $appObj->get("unit"), dbName => $dbName,
                        dbHost => $dbHost, dbPort => $appObj->get("dbPort"), dbType => $appObj->get("dbType"), height => $appObj->get("height"), width => $appObj->get("width"),
                        autorun => $appConfigObj->get("autorun"), wap => $appConfigObj->get("wapAware"));
    if ($companyAppObj->error)
    {
      myDie(error => $companyAppObj->errorMessage, configObj => $configObj);
    }
    my $result = $applicationObj->assignAppToCompany(companyAppObj => $companyAppObj);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if ($result == 1) # ok
    {
      print "App assigned to Company (" . $companyObj->get("name") . "), ID = '" . $companyObj->get("id") . "'.\n";
    }
    elsif ($result == -1) # app doesn't exist
    {
      myDie(error => "App no longer exists when assigning to Company (" . $companyObj->get("name") . "), ID = '" . $companyObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -2)  # company doesn't exist
    {
      myDie(error => "Company no longer exists when assigning to Company (" . $companyObj->get("name") . "), ID = '" . $companyObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -3)  # assignment already exists
    {
      print "App already assigned to Company (" . $companyObj->get("name") . "), ID = '" . $companyObj->get("id") . "'.\n";
    }
  }

  # now see if we need to assign to any users.
  my @Users = ();
  if ($appConfigObj->get("assignToAllUsers"))
  {
    if ($appConfigObj->get("appType") eq "administration" && $appConfigObj->get("appAdminType") eq "system")
    {
      myDie(error => "Cannot assign to all users when the app has an appAdminType of 'system'!\n", configObj => $configObj);
    }
    @Users = $authObj->getAllUsersInPortal;
  }
  elsif ($appConfigObj->get("assignToAdminUsers"))
  {
    @Users = $authObj->getAllUsersInPortal(admin => 1);
  }
  elsif ($appConfigObj->get("assignToSysAdminUsers"))
  {
    @Users = $authObj->getAllUsersInPortal(admin => 1, sysadmin => 1);
  }
  if ($authObj->error)
  {
    myDie(error => $authObj->errorMessage, configObj => $configObj);
  }

  # assign to each user the app.  Use the companyId from the userObj when building the UserApplicationObject.
  foreach my $userObj (@Users)
  {
    my $userAppObj = Portal::Objects::UserApplicationObject->new(langObj => $langObj);
    if ($userAppObj->error)
    {
      myDie(error => $userAppObj->errorMessage, configObj => $configObj);
    }
    $userAppObj->populate(userId => $userObj->get("id"), companyId => $userObj->get("companyId"), appId => $appObj->get("id"),
                     height => $appObj->get("height"), width => $appObj->get("width"), autorun => $appConfigObj->get("autorun"), wap => $appConfigObj->get("wapAware"));
    if ($userAppObj->error)
    {
      myDie(error => $userAppObj->errorMessage, configObj => $configObj);
    }
    # assign the app to the user.
    my $result = $applicationObj->assignAppToUser(userAppObj => $userAppObj);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if ($result == 1) # ok
    {
      print "\tApp assigned to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n";
    }
    elsif ($result == -1)  # app doesn't exist
    {
      myDie(error => "App no longer exists when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -2)  # company doesn't exist
    {
      myDie(error => "Company no longer exists when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -3)  # user doesn't exist
    {
      myDie(error => "User no longer exists when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -4)  # assignment already done
    {
      print "\tApp already assigned to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n";
    }
    elsif ($result == -5)  # app not assigned to company
    {
      myDie(error => "App not assigned to Company = '" . $userObj->get("companyId") . "' when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -6)  # license allotment violated
    {
      myDie(error => "License Allotment Violated when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
    elsif ($result == -7)  # app is administration and user is not an admin
    {
      myDie(error => "App is administration and user is not an admin when assigning to User (" . $userObj->get("fname") . " " . $userObj->get("lname") . "), ID = '" . $userObj->get("id") . "'.\n", configObj => $configObj);
    }
  }
}

print "\nappinstall.pl execution complete...\n\n";
exit 0;

# getInput - prompt, mask, and value
sub getInput
{
  my %args = (prompt => "", mask => "/.*/", value => "", @_);
  my $prompt = $args{prompt};
  my $mask = $args{mask};
  my $value = $args{value};

  my $input = "";
  my $done = 0;
  while (!$done)
  {
    print "\n$prompt [$value]: ";
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $value;
    }
    my $result;
    eval("\$result = \$input !~ $mask");
    if ($@)
    {
      die "Invalid eval on mask\nmask:$mask\input:$input\nresult:$result\n" . $@ . "\n";
    }
    elsif ($result == 1)
    {
      print "'$input' is invalid!\n";
      $input = "";
    }
    else
    {
      $done = 1;  # This will let me out even if I'm entering an empty string, as long as that is valid.
    }
  }
  $input =~ s/@/\\@/g;  # make sure that any @'s are escaped.
  $input =~ s/%/\\%/g;  # make sure that any %'s are escaped.
  return $input;
}

# myDie - Takes lang, encoding, error, configObj
sub myDie
{
  my %args = ( lang => 'en', encoding => 'iso-8859-1', error => "", configObj => undef, @_ );
  my $lang = $args{lang};
  my $encoding = $args{encoding};
  my $message = $args{error};
  my $configObj = $args{configObj};
  my $dateStamp = $methods->getCurrentDate(format => "%F %T");

  print("Error Occurred!\n");
  print($message);
  print("Have the Administrator check the error log ($dateStamp).\n");
  if (defined $portalDB)
  {
    my $logObj = Portal::Log->new(dbHandle => $portalDB, langObj => $langObj);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
    my $hostname = `hostname -i`;
    chomp $hostname;
    $hostname =~ s/ //g;

    my $logEntry = Portal::Objects::LogEntry->new(action => 18, ipAddress => $hostname, extraInfo => $message, userId => 0, langObj => $langObj);
    if ($logEntry->error)
    {
      die $logEntry->errorMessage;
    }
    $logObj->newEntry(logEntry => $logEntry);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
  }
  exit 1;
}
