#! /usr/bin/perl
# This script will install the specified applications colorScheme entries into the Portal.

# colorSchemeInstall.pl - Created by James Pattie, (james@pcxperience.com)
# Copyright (c) 2002-2003, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 12/26/2002

use Getopt::Long;
use Portal::Language;
use Portal::Data::Config;
use Portal::Auth;
use Portal::Log;
use Portal::Application;
use Portal::Objects::ApplicationObject;
use Portal::Data::Variables;
use Portal::Data::ColorScheme;
use Portal::Methods;
use strict;

# output a header so they at least have a clue what is going on in the
# postinst steps of the Debian packages.
print "colorSchemeInstall: ";

my $syntax = "colorSchemeInstall.pl --appname appName\n";
my $configVersion = "0.1";  # version of colorScheme XML config file.

# get the command line info so we know what application we are trying to assign to the Portal.
my $appName = "";
GetOptions('appname=s' => \$appName);

if (length $appName == 0)
{
  die "Error:  You did not specify the appname to use!\n$syntax";
}

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = undef;
eval "\$configObj = Portal::Data::Config->new(langObj => \$langObj);";
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($configObj->error)
{
  die $configObj->errorMessage;
}

my $methods = Portal::Methods->new(langObj => $langObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}

# make connection to the portal database
my $portalDB = $methods->portalDBSetup(type => "portal", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($portalDB->error)
{
  myDie(error => $portalDB->errorMessage, configObj => $configObj);
}

# setup a sessionObj to work with.
my $session = $methods->portalSession(portalDB => $portalDB, sessionId => "",
                                              configObj => $configObj, mode => "write");
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($session->error)
{
  myDie(error => $session->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Application module.
my $applicationObj = Portal::Application->new(portalDB => $portalDB, sessionObj => $session, langObj => $langObj);
if ($applicationObj->error)
{
  myDie(error => $applicationObj->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Auth module.
my $authObj = Portal::Auth->new(portalDB => $portalDB, sessionObj => $session, langObj => $langObj);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

# ask for the system admin account to use.

ADMIN:
print "What system administrator account should I use [admin]: ";
my $admin = getInput("", "admin");

# validate the username specified.
my $userObj = $authObj->getUserInfo(uname => $admin);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}
if (defined $userObj)
{
  if ($userObj->error)
  {
    myDie(error => $userObj->errorMessage, configObj => $configObj);
  }
}
else
{
  print "User '$admin' not found!  Retry...\n";
  goto ADMIN;
}

# now verify they are a system administrator
if (!$userObj->{sysadmin})
{
  print "User '$admin' is not a system administrator!  Retry...\n";
  goto ADMIN;
}
$session->{store}->{userObj} = $userObj;  # store the userObj in the session.
$session->{store}->{changed} = 1;


# verify that the specified app has colorScheme files defined in webRoot/install/appName/colorSchemes/
my $dir = $configObj->{webRoot} . $configObj->{siteDir} . "/install/$appName/colorSchemes";

if (! -d $dir)
{
  myDie(error => "$dir does not exist!\n", configObj => $configObj);
}
my @files = glob("$dir/*");
print "Processing " . int(scalar(@files)) . " colorScheme files...\n\n";
foreach my $file (@files)
{
  next if ($file !~ /\.xml$/);  # ignore any files not ending in .xml.

  (my $name = $file) =~ s/\.xml$//;  # strip off the .xml
  $name =~ s/^($dir\/)//;  # strip off the directory name.
  # turn any _ into spaces
  $name =~ s/_/ /g;
  print "Importing colorScheme '$name'...\n";

  # check and see if it already exists
  my $colorScheme = $applicationObj->getColorScheme(name => $name, userId => "-1");

  if ($applicationObj->error)
  {
    myDie(error => $applicationObj->errorMessage, configObj => $configObj);
  }

  # parse the colorScheme xml file.
  my $cfgObj = undef;
  eval { $cfgObj = Portal::XML::ConfigParser->new(langObj => $langObj); };
  if ($@)
  {
    myDie(error => "Instantiation of ConfigParser failed!<br>\n$@", configObj => $configObj);
  }
  my $settingsObj = undef;
  eval { $settingsObj = $cfgObj->parse(file => $file, module => "PortalColorScheme", version => $configVersion); };
  if ($@)
  {
    myDie(error => "Parse of '$file' failed!<br>\n$@", configObj => $configObj);
  }

  if (! defined $colorScheme)
  {
    # if not, then create after removing any 'update remove entries'
    foreach my $entry (keys %{$settingsObj->{settings}})
    {
      # see if this entry is one to delete.
      if ($entry =~ /^-/)
      {
        delete $settingsObj->{settings}->{$entry};
      }
    }

    my $string = "";
    eval { $string = $settingsObj->generateXML(); };
    if ($@)
    {
      myDie(error => $@, configObj => $configObj);
    }

    $colorScheme = Portal::Data::ColorScheme->new(name => $name, string => $string, userId => "-1", derivedFrom => "", langObj => $langObj);
    if ($colorScheme->error)
    {
      myDie(error => $colorScheme->errorMessage, configObj => $configObj);
    }
    my $result = $applicationObj->createColorScheme(colorScheme => $colorScheme);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if ($result == -1)
    {
      myDie(error => "creating colorScheme $name failed:  Already created, but it wasn't there a second ago!\n", configObj => $configObj);
    }
    elsif ($result == -2)
    {
      myDie(error => "creating colorScheme $name failed:  Permission denied!\n", configObj => $configObj);
    }
    elsif ($result == 1)
    {
      print "created colorScheme $name...\n";
    }
  }
  else
  {
    if ($colorScheme->error)
    {
      myDie(error => $colorScheme->errorMessage, configObj => $configObj);
    }
    # else, merge in and check for any derived color schemes and merge as needed.
    foreach my $entry (keys %{$settingsObj->{settings}})
    {
      # see if this entry is one to delete.
      if ($entry =~ /^-/)
      {
        # fixup and delete, if it exists.
        (my $tmpEntry = $entry) =~ s/^-//;
        if (exists $colorScheme->{$tmpEntry})
        {
          delete $colorScheme->{$tmpEntry};
          remove(colorScheme => $colorScheme, entry => $tmpEntry);
        }
        # now to remove the fgColor/bgColor entries.
        $tmpEntry =~ s/CSS$//;
        foreach my $tmpEntry2 (qw(fgColor bgColor))
        {
          my $tmpEntry3 = $tmpEntry . $tmpEntry2;
          if (exists $colorScheme->{$tmpEntry3})
          {
            delete $colorScheme->{$tmpEntry3};
            remove(colorScheme => $colorScheme, entry => $tmpEntry3);
          }
        }
      }
      else
      {
        # update
        if (! exists $colorScheme->{$entry})
        {
          push @{$colorScheme->{'_names_'}}, $entry;
        }
        $colorScheme->{$entry} = $settingsObj->{settings}->{$entry};
      }
    }

    # catchall code to make sure that we don't have any old fgColor/bgColor entries floating around.
    foreach my $entry (keys %{$settingsObj->{settings}})
    {
      if ($entry =~ /(fg|bg)Color$/)
      {
        delete $colorScheme->{$entry};
        remove(colorScheme => $colorScheme, entry => $entry);
      }
    }

    my $result = $applicationObj->updateColorScheme(colorScheme => $colorScheme);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    if ($result == -1)
    {
      myDie(error => "updating colorScheme $name failed:  Doesn't exist, but it was there a second ago!\n", configObj => $configObj);
    }
    elsif ($result == -2)
    {
      myDie(error => "updating colorScheme $name failed:  Permission denied!\n", configObj => $configObj);
    }
    elsif ($result == 1)
    {
      print "updated colorScheme $name...\n";
    }

    # now check for any derived entries and update.
    my %derived = $applicationObj->getDerivedColorSchemes(name => $name);
    if ($applicationObj->error)
    {
      myDie(error => $applicationObj->errorMessage, configObj => $configObj);
    }
    foreach my $tmpName (keys %derived)
    {
      my $tmpUserId = $derived{$tmpName};

      $colorScheme = $applicationObj->getColorScheme(name => $tmpName, userId => $tmpUserId);
      if ($applicationObj->error)
      {
        myDie(error => $applicationObj->errorMessage, configObj => $configObj);
      }
      if (! defined $colorScheme)
      {
        myDie(error => "colorScheme $tmpName for userId $tmpUserId does not exist!\n", configObj => $configObj);
      }
      else
      {
        if ($colorScheme->error)
        {
          myDie(error => $colorScheme->errorMessage, configObj => $configObj);
        }
      }

      foreach my $entry (keys %{$settingsObj->{settings}})
      {
        # see if this entry is one to delete.
        if ($entry =~ /^-/)
        {
          # fixup and delete, if it exists.
          (my $tmpEntry = $entry) =~ s/^-//;
          if (exists $colorScheme->{$tmpEntry})
          {
            delete $colorScheme->{$tmpEntry};
            remove(colorScheme => $colorScheme, entry => $tmpEntry);
          }
          # now to remove the fgColor/bgColor entries.
          $tmpEntry =~ s/CSS$//;
          foreach my $tmpEntry2 (qw(fgColor bgColor))
          {
            my $tmpEntry3 = $tmpEntry . $tmpEntry2;
            if (exists $colorScheme->{$tmpEntry3})
            {
              delete $colorScheme->{$tmpEntry3};
              remove(colorScheme => $colorScheme, entry => $tmpEntry3);
            }
          }
        }
        else
        {
          # update, only if it does not exist
          if (! exists $colorScheme->{$entry})
          {
            $colorScheme->{$entry} = $settingsObj->{settings}->{$entry};
            push @{$colorScheme->{'_names_'}}, $entry;
          }
        }
      }

      # catchall code to make sure that we don't have any old fgColor/bgColor entries floating around.
      foreach my $entry (keys %{$settingsObj->{settings}})
      {
        if ($entry =~ /(fg|bg)Color$/)
        {
          delete $colorScheme->{$entry};
          remove(colorScheme => $colorScheme, entry => $entry);
        }
      }

      my $result = $applicationObj->updateColorScheme(colorScheme => $colorScheme);
      if ($applicationObj->error)
      {
        myDie(error => $applicationObj->errorMessage, configObj => $configObj);
      }
      if ($result == -1)
      {
        myDie(error => "updating colorScheme $tmpName for userId $tmpUserId failed:  Doesn't exist, but it was there a second ago!\n", configObj => $configObj);
      }
      elsif ($result == -2)
      {
        myDie(error => "updating colorScheme $tmpName for userId $tmpUserId failed:  Permission denied!\n", configObj => $configObj);
      }
      elsif ($result == 1)
      {
        print "updated colorScheme $tmpName for userId $tmpUserId...\n";
      }
    }
  }
}

# cleanup the session.
$session->delete;

print "\ncolorSchemeInstall.pl execution complete...\n";
exit 0;

# myDie - Takes lang, encoding, error, configObj
sub myDie
{
  my %args = ( lang => 'en', encoding => 'iso-8859-1', error => "", configObj => undef, @_ );
  my $lang = $args{lang};
  my $encoding = $args{encoding};
  my $message = $args{error};
  my $configObj = $args{configObj};
  my $dateStamp = $methods->getCurrentDate(format => "%F %T");

  print("colorSchemeInstall: Error Occurred!\n");
  print($message);
  print("Have the Administrator check the error log ($dateStamp).\n");
  if (defined $portalDB)
  {
    my $logObj = Portal::Log->new(dbHandle => $portalDB, langObj => $langObj);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
    my $hostname = `hostname -i`;
    chomp $hostname;
    $hostname =~ s/ //g;

    my $logEntry = Portal::Objects::LogEntry->new(action => 18, ipAddress => $hostname, extraInfo => $message, userId => 0, langObj => $langObj);
    if ($logEntry->error)
    {
      die $logEntry->errorMessage;
    }
    $logObj->newEntry(logEntry => $logEntry);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
  }
  exit 1;
}

# string getInput(string, default)
# pass in any initial values you want or an empty string and the default value to use
# if the user presses Enter.  You get back what was either passed in or what the user entered.
sub getInput
{
  my $input = shift;
  my $default = shift;
  while (length $input == 0)
  {
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $default;
    }
  }
  return $input;
}

# void remove(colorScheme, entry)
# removes the entry from _names_ array in the specified colorScheme.
sub remove
{
  my %args = ( colorScheme => undef, entry => "", @_ );
  my $colorScheme = $args{colorScheme};
  my $entry = $args{entry};

  for (my $i=0; $i < scalar(@{$colorScheme->{'_names_'}}); $i++)
  {
    if ($colorScheme->{'_names_'}[$i] eq $entry)
    {  # delete it
      #print "Found '$entry' at '$i'...\n";
      splice(@{$colorScheme->{'_names_'}}, $i, 1);
      last;  # jump out.
    }
  }
}
