#! /usr/bin/perl
# This script will convert all ColorScheme XML files in the current directory into the 
# consolidated ColorScheme CSS format that JT has moved me towards.

# convertColorSchemeXML.pl - Created by James Pattie, (james@pcxperience.com)
# Copyright (c) 2002-2004, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 2004-12-03

use Portal::Language;
use Portal::Data::ColorScheme;
use strict;

print "convertColorSchemeXML: ";

my $syntax = "convertColorSchemeXML.pl\n";
my $configVersion = "0.1";  # version of colorScheme XML config file.

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

# verify that we have xml files to work with in the current directory.
my @files = glob("*.xml");
print "Processing " . int(scalar(@files)) . " colorScheme files...\n\n";
foreach my $file (@files)
{
  (my $name = $file) =~ s/\.xml$//;  # strip off the .xml
  # turn any _ into spaces
  $name =~ s/_/ /g;
  print "Converting colorScheme '$name'...\n";

  # parse the colorScheme xml file.
  my $cfgObj = undef;
  eval { $cfgObj = Portal::XML::ConfigParser->new(langObj => $langObj); };
  if ($@)
  {
    myDie(error => "Instantiation of ConfigParser failed!<br>\n$@");
  }
  my $settingsObj = undef;
  eval { $settingsObj = $cfgObj->parse(file => $file, module => "PortalColorScheme", version => $configVersion); };
  if ($@)
  {
    myDie(error => "Parse of '$file' failed!<br>\n$@");
  }

  foreach my $entry (keys %{$settingsObj->{settings}})
  {
    # see if this entry is not one already marked to delete.
    if ($entry !~ /^-/)
    {
      # skip the bg/fgColor entries for now.
      next if ($entry =~ /(bg|fg)Color$/);
      
      # store the current CSS style sheet entry.
      my $value = $settingsObj->{settings}->{$entry};
      
      # remove the CSS from the entry name.
      (my $tmpEntry = $entry) =~ s/CSS$//;
      
      # get the colors.
      $value = "color: " . $settingsObj->{settings}->{$tmpEntry."fgColor"} . "; background-color: " . $settingsObj->{settings}->{$tmpEntry."bgColor"} . ";" . (length $value > 0 ? " " . $value : "");
      
      # delete the old entries.
      foreach my $tmpEntry2 (qw(CSS fgColor bgColor))
      {
        my $tmpEntry3 = $tmpEntry . $tmpEntry2;
        if (exists $settingsObj->{settings}->{$tmpEntry3})
        {
          delete $settingsObj->{settings}->{$tmpEntry3};
        }
      }
      
      # generate the "delete" entry.
      $settingsObj->{settings}->{"-".$tmpEntry."CSS"} = "";
      
      # now generate the new entry, after removing the ()'s.
      $tmpEntry =~ s/[()]//g;
      $settingsObj->{settings}->{$tmpEntry} = $value;
    }
  }
  
  # now generate the new converted config file.
  my $string = "";
  eval { $string = $settingsObj->generateXML(); };
  if ($@)
  {
    myDie(error => $@);
  }
  
  open FILE, ">$file.new" or die "Couldn't create $file.new!  $!";
  print FILE $string;
  close FILE;
  
  print "Created $file.new...\n";
}

print "\nconvertColorSchemeXML.pl execution complete...\n";
exit 0;

# myDie - Takes lang, encoding, error, configObj
sub myDie
{
  my %args = ( error => "", @_ );
  my $message = $args{error};

  print("Error Occurred!\n");
  print($message);
  exit 1;
}
