#! /usr/bin/perl
# This script will create the default company and administrative account
# for the Portal system to use.

# create_default_account.pl - Created by James A. Pattie, (james@pcxperience.com)
# Copyright (c) 2000-2003, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 11/24/2000

use Portal::Language;
use Portal::Data::Config;
use Portal::Auth;
use Portal::Application;
use Portal::Log;
use Portal::Objects::UserPreferenceObject;
use Portal::Methods;
use strict;

$| = 1;

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = undef;
eval { $configObj = Portal::Data::Config->new(langObj => $langObj); };
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($configObj->error)
{
  die $configObj->errorMessage;
}

my $methods = Portal::Methods->new(langObj => $langObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}

# make connection to the portal database
my $portalDB = $methods->portalDBSetup(type => "portal", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($portalDB->error)
{
  myDie(error => $portalDB->errorMessage, configObj => $configObj);
}

my $billingDB = $methods->portalDBSetup(type => "billing", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($billingDB->error)
{
  myDie(error => $billingDB->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Auth module.
my $authObj = Portal::Auth->new(portalDB => $portalDB, billingDB => $billingDB, langObj => $langObj);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

my $applicationObj = Portal::Application->new(portalDB => $portalDB, langObj => $langObj);
if ($applicationObj->error)
{
  myDie(error => $applicationObj->errorMessage, configObj => $configObj);
}

# check and make sure no companies are defined.
my @Companies = $authObj->getListOfCompanies;
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}
if (scalar @Companies > 0)
{
  print "There is at least one company already created!\n";
  exit 0;
}

# now we prompt the user for the company info to create.
my %companyAttributes = ();
my %companyBillingAttributes = ();
my %userAttributes = ();
my %userPreferences = ();

COMPANY_INFO:
print "\nEnter the Info for the Default Company:\n\n";

$companyAttributes{parentId} = -1;  # We are a parent company.
$companyAttributes{country} = getInput(prompt => "Country Code (2 letter value)", value => "US", mask => "[A-Z]{2}");
$companyAttributes{name} = getInput(prompt => "Name", value => "", mask => ".+");
$companyAttributes{code} = getInput(prompt => "Code", value => "", mask => ".+");
$companyAttributes{address1} = getInput(prompt => "Address1", value => "", mask => ".+");
$companyAttributes{address2} = getInput(prompt => "Address2", value => "");
$companyAttributes{city} = getInput(prompt => "City", value => "", mask => ".+");
$companyAttributes{state} = getInput(prompt => "State", value => "", mask => "[A-Z]{2}");
$companyAttributes{zip} = getInput(prompt => "Zip", value => "", mask => "\\d{5}(-\\d{4})?");
$companyAttributes{phone} = getInput(prompt => "Phone", value => "", mask => "\\d{3}-\\d{3}-\\d{4}");
$companyAttributes{fax} = getInput(prompt => "Fax", value => "", mask => "(\\d{3}-\\d{3}-\\d{4})?");
$companyAttributes{email} = getInput(prompt => "E-mail", value => "", mask => ".+\\@[\\w-]+\\.\\D+");
$companyAttributes{comment} = getInput(prompt => "Comment", value => "", mask => ".*");
$companyAttributes{language} = getInput(prompt => "Language", value => "en", mask => "[a-z]{2}");
$companyAttributes{timeFormat} = getInput(prompt => "Time Format (12=AM/PM, 24=Military Time)", value => "12", mask => "(12|24)");
$companyAttributes{tz} = getInput(prompt => "Time Zone", value => "CST", mask => "[A-Z]{3}");
$companyAttributes{active} = 1;
$companyAttributes{url} = getInput(prompt => "Your website url", value => "http://www.xyz.com/", mask => "https?:\/\/[^\/]+\.[^\/]+\/?");
$companyAttributes{logoUrl} = getInput(prompt => "Your logo's url", value => "$companyAttributes{url}" . ($companyAttributes{url} =~ /\/$/ ? "" : "/") . "images/logo.png", mask => "https?:\/\/[^\/]+\.[^\/]+\/.+\.(gif|png|jpg|tif|bmp)");
$companyAttributes{billingMethod} = getInput(prompt => "What billing method (n=none, c=credit card, i=invoice)?", value => "n", mask => "(n|c|i)");

# create the company.
my $companyObj = Portal::Objects::CompanyObject->new(%companyAttributes, langObj => $langObj);
if ($companyObj->error)
{ # invalid company info
  print $companyObj->errorMessage . "\n";
  goto COMPANY_INFO; # restart
}
# now create the company.
if (!$authObj->createCompany(companyObj => $companyObj))
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

# get the id for the newly created company.
$companyObj = $authObj->getCompanyInfo(code => $companyAttributes{code});
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

COMPANY_BILLING_INFO:
if ($companyObj->{billingMethod} eq "c")
{
  print "\nEnter the Credit Card Billing Info for '$companyAttributes{name}':\n\n";

  $companyBillingAttributes{id} = $companyObj->id;
  $companyBillingAttributes{ccType} = getInput(prompt => "Credit Card Type", value => "Visa");
  $companyBillingAttributes{ccNum} = getInput(prompt => "Credit Card Number", value => "1234567890123456", mask => "\\d{16}");
  $companyBillingAttributes{ccExpire} = getInput(prompt => "Expiration Date", value => "01/01/2003", mask => "\\d{2}(-|/)\\d{2}(-|/)\\d{4}");
}
else
{
  print "\nCreating Default Billing Info...\n\n";
  $companyBillingAttributes{id} = $companyObj->id;
  $companyBillingAttributes{ccType} = "Visa";
  $companyBillingAttributes{ccNum} = "1234567890123456";
  $companyBillingAttributes{ccExpire} = "01/01/2003";
}
my $companyBillObj = Portal::Objects::CompanyBillingObject->new(%companyBillingAttributes, langObj => $langObj);
if ($companyBillObj->error)
{
  print $companyBillObj->errorMessage . "\n";
  goto COMPANY_BILLING_INFO;
}
if (!$authObj->createCompanyBilling(companyBillObj => $companyBillObj))
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

USER_INFO:
# prompt the user for the user info to create.

print "\nEnter Info for System Administrative User Account in Company '$companyObj->{name}':\n\n";

$userAttributes{companyId} = $companyObj->id;
$userAttributes{uname} = getInput(prompt => "User Name", value => "admin");
$userAttributes{fname} = getInput(prompt => "First Name", value => "System");
$userAttributes{mname} = getInput(prompt => "Middle Name", value => "", mask => ".*");
$userAttributes{lname} = getInput(prompt => "Last Name", value => "Administrator");
$userAttributes{email} = getInput(prompt => "E-mail", value => "", mask => ".+\\@[\\w-]+\\.\\D+");
PASSWORD:
  my $pass1 = getInput(prompt => "Password", value => "", mask => ".{6,}");
  my $pass2 = getInput(prompt => "Re-Enter Password", value => "", mask => ".{6,}");
  if ($pass1 ne $pass2)
  {
    print "Passwords do not equal!  Retry.\n";
    goto PASSWORD;
  }
$userAttributes{password} = $pass1;
$userAttributes{passCrypted} = 0;  # indicate the password needs to be crypted.
$userAttributes{empId} = getInput(prompt => "Employee ID", value => "", mask => ".*");
$userAttributes{ssn} = getInput(prompt => "SSN", value => "999999999", mask => "\\d{9}");
$userAttributes{language} = getInput(prompt => "Language", value => "en", mask => "[a-z]{2}");
$userAttributes{timeFormat} = getInput(prompt => "Time Format (12=AM/PM, 24=Military Time)", value => "12", mask => "(12|24)");
$userAttributes{tz} = getInput(prompt => "Time Zone", value => "CST", mask => "[A-Z]{3}");
$userAttributes{comment} = getInput(prompt => "Comment", value => "Portal System Administrator.", mask => ".*");
$userAttributes{active} = 1;
$userAttributes{admin} = 1;
$userAttributes{sysadmin} = 1;
print "\nUser Preference Settings:\n";
$userPreferences{Desktop}{dynamicContent} = getInput(prompt => "Dynamic Content on Desktop", value => "n", mask => "(y|n|Y|N)");
$userPreferences{Desktop}{dynamicContent} = ($userPreferences{Desktop}{dyanmicContent} =~ /^Y$/i ? 1 : 0);

$userPreferences{Desktop}{dynamicContentConfig} = getConfigDefault(app => "Portal", module => "Desktop", prefName => "dynamicContentConfig_Default");
$userPreferences{Global}{colorScheme} = getConfigDefault(app => "Portal", module => "Global", prefName => "colorScheme_Default");

# create the user.
my $userObj = Portal::Objects::User->new(langObj => $langObj);
$userObj->populate(%userAttributes) if (!$userObj->error);
if ($userObj->error)
{
  print $userObj->errorMessage . "\n";
  goto USER_INFO;
}
if (!$authObj->createUser(userObj => $userObj))
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}
$userObj = $authObj->getUserInfo(uname => $userAttributes{uname});
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

print "\n\nCompany (" . $companyObj->name . ", ID=" . $companyObj->id . ") created.\n";
print "User (" . $userObj->uname . ", ID=" . $userObj->id . ") created.\n";
print "Just install the applications now.\n";

# create any preferences needed.
foreach my $module (keys %userPreferences)
{
  foreach my $pref (keys %{$userPreferences{$module}})
  {
    my $userPreferenceObj = Portal::Objects::UserPreferenceObject->new(id => $userObj->{id}, app => "Portal", name => "$pref", value => $userPreferences{$module}{$pref}, module => $module, langObj => $langObj);
    if ($userPreferenceObj->error)
    {
      die $userPreferenceObj->errorMessage;
    }
    my $result = $authObj->createUserPreference(userPreferenceObj => $userPreferenceObj);
    if ($authObj->error)
    {
      die $authObj->errorMessage;
    }
    if ($result == -1)
    {
      die "Somehow user $userObj->{uname} has already set preference: " . $userPreferenceObj->print . "\n";
    }
    elsif ($result == -2)
    {
      die "App = 'Portal' does not exist!\nThis error should never happen!\n";
    }
    else
    {
      print "Successfully made preference: " . $userPreferenceObj->print . "\n";
    }
  }
}

exit 0;

# getInput - prompt, mask, and value
sub getInput
{
  my %args = (prompt => "", mask => ".*", value => "", @_);
  my $prompt = $args{prompt};
  my $mask = $args{mask};
  my $value = $args{value};

  my $input = "";
  my $done = 0;
  while (!$done)
  {
    print "\n$prompt [$value]: ";
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $value;
    }
    if ($input !~ /^$mask$/)
    {
      print "'$input' is invalid!\n";
      $input = "";
    }
    else
    {
      $done = 1;  # This will let me out even if I'm entering an empty string, as long as that is valid.
    }
  }
  $input =~ s/@/\\@/g;  # make sure that any @'s are escaped.
  $input =~ s/%/\\%/g;  # make sure that any %'s are escaped.
  return $input;
}

# myDie - Takes lang, encoding, error, configObj
sub myDie
{
  my %args = ( lang => 'en', encoding => 'iso-8859-1', error => "", configObj => undef, @_ );
  my $lang = $args{lang};
  my $encoding = $args{encoding};
  my $message = $args{error};
  my $configObj = $args{configObj};
  my $dateStamp = $methods->getCurrentDate(format => "%F %T");

  print("Error Occurred!\n");
  print($message);
  print("Have the Administrator check the error log ($dateStamp).\n");
  my $logObj = Portal::Log->new(dbHandle => $portalDB, langObj => $langObj);
  if ($logObj->error)
  {
    die $logObj->errorMessage;
  }
  my $hostname = `hostname -i`;
  chomp $hostname;
  $hostname =~ s/ //g;

  my $logEntry = Portal::Objects::LogEntry->new(action => 18, ipAddress => $hostname, extraInfo => $message, userId => 0, langObj => $langObj);
  if ($logEntry->error)
  {
    die $logEntry->errorMessage;
  }
  $logObj->newEntry(logEntry => $logEntry);
  if ($logObj->error)
  {
    die $logObj->errorMessage;
  }
  exit 1;
}

# scalar getConfigDefault(app, module, prefName)
sub getConfigDefault
{
  my %args = (app => "", module => "", prefName => "", @_);
  my $callingApp = $args{app};
  my $prefModule = $args{module};
  my $prefName = $args{prefName};

  # get the default for this preference
  my $prefName = $callingApp . "_" . $prefModule . "_" . $prefName;
  my $prefValue = $methods->getConfigValue(name => $prefName, portalDB => $applicationObj->{portalDB});
  if ($methods->error)
  {
    myDie(error => $methods->errorMessage, configObj => $configObj);
  }
  if (!defined $prefValue)
  {
    myDie(error => "config entry name = '$prefName' does not exist!\nYou need to upgrade the Database!\n", configObj => $configObj);
  }
  return $prefValue;
}
