#! /usr/bin/perl
# This script will create the directories and copy the images
# to the images directory of the webserver as configured in
# Portal::Data::Config.

# install_app.pl - Created by James Pattie, (james@pcxperience.com)
# Copyright (c) 2000-2003, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 08/08/2001
# 04/19/2002 - Updated to work with the new xml config files.
# 12/26/2002 - Added support for the colorSchemes files.
# 08/05/2003 - Added support for the misc directory.
# 2004-07-05 - Added support for appname-soap.cgi cgi-bin file.

use Getopt::Long;
use Portal::Language;

$| = 1;

my $syntax = "install_app.pl --appname appName --location installFilesLocation --configversion version --override 1|0 [--prefix rootPrefix]\n";

# get the command line info so we know what application we are trying to assign to the Portal.
my $prefix = "";
my $appName = "";
my $location = "";
my $configVersion = "";
my $override = "";
GetOptions('appname=s' => \$appName, 'prefix:s' => \$prefix, 'location=s' => \$location, 'configversion=f' => \$configVersion, 'override=i' => \$override);

if (length $prefix > 0)
{
  if (! -d $prefix)
  {
    die "Error!  Prefix directory = '$prefix' does not exist!\n$syntax";
  }
}

if (length $appName == 0)
{
  die "Error:  You did not specify the appname to use!\n$syntax";
}

if (length $location == 0)
{
  die "Error:  You did not specify the location to use!\n$syntax";
}
if (! -d $location)
{
  die "Error:  Location = '$location' does not exist!\n$syntax";
}

if ($configVersion !~ /^(\d+\.\d+)$/)
{
  die "Error:  configversion = '$configVersion' is invalid!\n$syntax";
}
if ($override !~ /^(0|1)$/)
{
  die "Error:  override = '$override' is invalid!\n$syntax";
}

use Portal::Data::Config;
use strict;

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = undef;
eval { $configObj = Portal::Data::Config->new(langObj => $langObj); };
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($configObj->error)
{
  die $configObj->errorMessage;
}

my $dir = $prefix . $configObj->webRoot . $configObj->siteDir;

if (length $prefix == 0)
{
  if (! -d $dir)
  { # we need to make it
    die "Error Portal web directory = '$dir' doesn't exist!\n";
  }

  # make sure the soap directory exists
  if (! -e "$dir/soap")
  {
    die "Error Portal soap directory = '$dir/soap' doesn't exist!\n";
  }

  # make sure the images directory exists
  if (! -e "$dir/images")
  {
    die "Error Portal images directory = '$dir/images' doesn't exist!\n";
  }

  if (! -e "$dir/images/$appName")
  {
    print "Creating '$dir/images/$appName'...\n";
    if (mkdir ("$dir/images/$appName", 0755) == 0)
    {
      die "Error creating '$dir/images/$appName'!  Error = $!\n";
    }
  }
  else
  {
    # remove old files
    system("/bin/rm -Rf $dir/images/$appName/*") == 0 or die "Error removing old files from '$dir/images/$appName'!  Error = $!\n";
  }

  # make sure the js directory exists
  if (! -e "$dir/js")
  {
    die "Error Portal js directory = '$dir/js' doesn't exist!\n";
  }

  # make sure the css directory exists
  if (! -e "$dir/css")
  {
    die "Error Portal css directory = '$dir/css' doesn't exist!\n";
  }

  # make sure the install directory exists
  if (! -e "$dir/install")
  {
    die "Error Portal install directory = '$dir/install' doesn't exist!\n";
  }

  # make sure the db_dump directory exists
  if (! -e "$dir/db_dump")
  {
    die "Error Portal db_dump directory = '$dir/db_dump' doesn't exist!\n";
  }

  # make sure the misc directory exists
  if (! -e "$dir/misc")
  {
    die "Error Portal misc directory = '$dir/misc' doesn't exist!\n";
  }

  # make sure the cron directory exists
  if (! -e "$prefix/usr/lib/pcx_portal/cron")
  {
    die "Error Portal cron directory = '$prefix/usr/lib/pcx_portal/cron' doesn't exist!\n";
  }
}
else
{
  if (! -d $dir)
  { # we need to make it
    print "Creating '$dir'...\n";
    if (system("/bin/mkdir -p -m 0755 $dir") != 0)
    {
      die "Error creating '$dir'!  Error = $!\n";
    }
  }

  # make sure the soap directory exists
  if (! -e "$dir/soap")
  {
    print "Creating '$dir/soap'...\n";
    if (mkdir ("$dir/soap", 0700) == 0)
    {
      die "Error creating '$dir/soap'!  Error = $!\n";
    }
  }

  # make sure the images directory exists
  if (! -e "$dir/images")
  {
    print "Creating '$dir/images'...\n";
    if (mkdir ("$dir/images", 0755) == 0)
    {
      die "Error creating '$dir/images'!  Error = $!\n";
    }
  }

  if (! -e "$dir/images/$appName")
  {
    print "Creating '$dir/images/$appName'...\n";
    if (mkdir ("$dir/images/$appName", 0755) == 0)
    {
      die "Error creating '$dir/images/$appName'!  Error = $!\n";
    }
  }

  # make sure the js directory exists
  if (! -e "$dir/js")
  {
    print "Creating '$dir/js'...\n";
    if (mkdir ("$dir/js", 0755) == 0)
    {
      die "Error creating '$dir/js'!  Error = $!\n";
    }
  }

  # make sure the css directory exists
  if (! -e "$dir/css")
  {
    print "Creating '$dir/css'...\n";
    if (mkdir ("$dir/css", 0755) == 0)
    {
      die "Error creating '$dir/css'!  Error = $!\n";
    }
  }

  # make sure the install directory exists
  if (! -e "$dir/install")
  {
    print "Creating '$dir/install'...\n";
    if (mkdir ("$dir/install", 0755) == 0)
    {
      die "Error creating '$dir/install'!  Error = $!\n";
    }
  }

  # make sure the db_dump directory exists
  if (! -e "$dir/db_dump")
  {
    print "Creating '$dir/db_dump'...\n";
    if (mkdir ("$dir/db_dump", 0700) == 0)
    {
      die "Error creating '$dir/db_dump'!  Error = $!\n";
    }
  }

  # make sure the misc directory exists
  if (! -e "$dir/misc")
  {
    print "Creating '$dir/misc'...\n";
    if (mkdir ("$dir/misc", 0700) == 0)
    {
      die "Error creating '$dir/misc'!  Error = $!\n";
    }
  }

  # make sure the cron directory exists
  if (! -e "$prefix/usr/lib/pcx_portal/cron")
  {
    print "Creating '$prefix/usr/lib/pcx_portal/cron'...\n";
    system("/bin/mkdir -p -m 0700 $prefix/usr/lib/pcx_portal/cron") == 0 or die "Error creating '$prefix/usr/lib/pcx_portal/cron'!  Error = $!\n";
  }
}

# copy the soap cgi script file(s)
print "Copying SOAP scripts...\n";
system("/bin/cp -Rf $location/files/soap/*.cgi $dir/soap/") == 0 or die "Error copying files to '$dir/soap/'!  Error = $!\n";

# copy the portal application image files
print "Copying image files...\n";
system("/bin/cp -Rf $location/files/images/* $dir/images/$appName") == 0 or die "Error copying files to '$dir/images/'!  Error = $!\n";

foreach my $files ("install", "js", "css", "misc")
{
  # see if the application has any specific files it needs
  if (-e "$location/files/$files")
  {
    # see if we need to make the directory under the $dir/$files directory.
    if (! -e "$dir/$files/$appName")
    {
      if (mkdir("$dir/$files/$appName", 0755) == 0)
      {
        die "Error creating '$dir/$files/$appName'!  Error = $!\n";
      }
    }
    else
    {
      print "'$dir/$files/$appName' already exists.  Skipping creation.\n";
      if ($files !~ /^(misc)$/)
      {
        # remove old files
        system("/bin/rm -Rf $dir/$files/$appName/*") == 0 or die "Error removing old files from '$dir/$files/$appName'!  Error = $!\n";
      }
    }

    # now copy the application specific files over
    print "Copying application specific $files files...\n";
    system("/bin/cp -Rf $location/files/$files/* $dir/$files/$appName") == 0 or die "Error copying files to '$dir/$files/$appName/'!  Error = $!\n";

    # set the ownership on the files so the Portal/WebServer can work with them.
    if ($files =~ /^(misc)$/)
    {
      system("chown -R $configObj->{webUser}:$configObj->{webGroup} $dir/$files/$appName") == 0 or die "Error changing ownership to $configObj->{webUser}:$configObj->{webGroup} for $dir/$files/$appName! $!\n";
    }
  }
}

# now take care of any colorScheme entries
foreach my $files ("colorSchemes")
{
  # see if the application has any specific files it needs
  if (-e "$location/files/$files")
  {
    # see if we need to make the directory under the $dir/install/$files directory.
    if (! -e "$dir/install/$appName")
    {
      if (mkdir("$dir/install/$appName", 0755) == 0)
      {
        die "Error creating '$dir/install/$appName'!  Error = $!\n";
      }
    }
    else
    {
      print "'$dir/install/$appName' already exists.  Skipping creation.\n";
    }
    if (! -e "$dir/install/$appName/$files")
    {
      if (mkdir("$dir/install/$appName/$files", 0755) == 0)
      {
        die "Error creating '$dir/install/$appName/$files'!  Error = $!\n";
      }
    }
    else
    {
      print "'$dir/install/$appName/$files' already exists.  Skipping creation.\n";
      # remove old files
      system("/bin/rm -Rf $dir/install/$appName/$files/*") == 0 or die "Error removing old files from '$dir/install/$appName/$files'!  Error = $!\n";
    }

    # now copy the application specific files over
    print "Copying application specific $files files...\n";
    system("/bin/cp -Rf $location/files/$files/* $dir/install/$appName/$files") == 0 or die "Error copying files to '$dir/install/$appName/$files/'!  Error = $!\n";
  }
}

# now handle any cron files the app needs
my @cronValues = ("hourly", "daily", "weekly", "monthly");
if (-d "$location/files/cron")
{
  foreach my $cronDir (@cronValues)
  {
    if (! -d "$prefix/etc/cron.$cronDir")
    {
      # create the /etc/cron.$cronDir directory.  Only needed for rpm installs in a prefix environment.
      print "Creating '$prefix/etc/cron.$cronDir'...\n";
      system("/bin/mkdir -p -m 0755 $prefix/etc/cron.$cronDir") == 0 or die "Error:  Couldn't create '$prefix/etc/cron.$cronDir' directory!  $!\n";
    }

    if (! -d "$prefix/usr/lib/pcx_portal/cron/$appName/$cronDir")
    {
      # create the /usr/lib/pcx_portal/cron/$appName/$cronDir directory.
      print "Creating '$prefix/usr/lib/pcx_portal/cron/$appName/$cronDir'...\n";
      system("/bin/mkdir -p -m 0755 $prefix/usr/lib/pcx_portal/cron/$appName/$cronDir") == 0 or die "Error:  Couldn't create '$prefix/usr/lib/pcx_portal/cron/$appName/$cronDir' directory!  $!\n";
    }

    # copy the cron scripts into place
    my @files = glob("files/cron/$cronDir/*");
    foreach my $file (@files)
    {
      $file =~ s/^(files\/cron\/$cronDir\/)//g;
      system("cp -f files/cron/$cronDir/$file $prefix/usr/lib/pcx_portal/cron/$appName/$cronDir/$file") == 0 or die "Error copying 'files/cron/$cronDir/$file' to '$prefix/usr/lib/pcx_portal/cron/$appName/$cronDir/$file'!\n";
    }
  }

  foreach my $cronDir (@cronValues)
  {
    my @files = glob("files/cron/$cronDir/*");
    next if (scalar @files == 0);
    foreach my $file (@files)
    {
      $file =~ s/^(files\/cron\/$cronDir\/)//g;
      print "\t$cronDir: $file\n";
    }
    print "\nDo you want to link to these Cron Jobs? [Y]\n";
    my $input = "";
    $input = getInput($input, "y");
    if ($input =~ /^y$/i)
    {
      foreach my $file (@files)
      {
        $file =~ s/^(files\/cron\/$cronDir\/)//g;
        system("rm -f $prefix/etc/cron.$cronDir/$file") == 0 or die "Error removing old file/link $prefix/etc/cron.$cronDir/$file! $!\n";
        system("ln -s /usr/lib/pcx_portal/cron/$appName/$cronDir/$file $prefix/etc/cron.$cronDir/$file") == 0 or die "Error linking $prefix/usr/lib/pcx_portal/cron/$appName/$cronDir/$file to $prefix/etc/cron.$cronDir/$file! $!\n";
      }
    }
  }
}

# first of all, we have to make sure the program is configured!
# copy the $appName.xml file from config to /etc/pcx_portal/$appName.xml.  Backup any files already there.
use File::Copy;

# see if we need to convert their old $appName/Config.pm to the new $appName.xml config format.
if ( -e "$prefix/etc/pcx_portal/$appName/Config.pm" )
{
  system("/usr/lib/pcx_portal/convertConfig.pl $appName $prefix/etc/pcx_portal/$appName/Config.pm $prefix/etc/pcx_portal/$appName.xml") == 0 or die "Error converting old config file for $appName!  Error = $!\n";
  # now remove the old config directory.
  system("/bin/rm -rf $prefix/etc/pcx_portal/$appName") == 0 or die "Error removing old Config directory for $appName!  Error = $!\n";
}

if ( -e "$prefix/etc/pcx_portal/$appName.xml" )
{
  # pull in the settings from the config file.
  my $configFile = "$prefix/etc/pcx_portal/$appName.xml";
  my $cutCmd = `which cut`;
  chomp $cutCmd;
  my $version = `/bin/cat $configFile | /bin/grep config | /bin/grep version | $cutCmd -d \" \" -f3 | $cutCmd -d \"\\\"\" -f2`;
  chomp $version;

  $override = 0 if ($version eq $configVersion);

  if ($override)
  {
    print "Backing up $appName.xml file...\n";
    copy("$prefix/etc/pcx_portal/$appName.xml", "$prefix/etc/pcx_portal/$appName.xml.bak") or die "Error backing up $appName.xml!  Error = $!\n";
  }
}

if (! -d "$prefix/etc/pcx_portal")
{
  # create the config directory
  print "Creating '$prefix/etc/pcx_portal'...\n";
  system("/bin/mkdir -p -m 0755 $prefix/etc/pcx_portal") == 0 or die "Error creating '$prefix/etc/pcx_portal'!  Error = $!\n";
}
else
{
  print "'$prefix/etc/pcx_portal' already exists.  Skipping creation.\n";
}

# validate we have the config directory with $appName.xml in it.
if (! -d "$location/config")
{
  die "Error!  You do not have the config directory!\n";
}
if (! -e "$location/config/$appName.xml")
{
  die "Error!  You do not have the $appName.xml file!\n";
}

# now copy the $appName.xml file from config to /etc/pcx_portal/$appName.xml
if ($override)
{
  print "Copying $appName.xml file to '$prefix/etc/pcx_portal/'...\n";
  copy("$location/config/$appName.xml", "$prefix/etc/pcx_portal/$appName.xml") or die "Error copying $appName.xml to $prefix/etc/pcx_portal/!  Error = $!\n";
}

# now run setup.pl to actually configure the Portal App before we allow them to try and copy files around.
if (! -e "$location/scripts/setup.pl")
{
  die "Error!  setup.pl does not exist!\n";
}
if (! -x "$location/scripts/setup.pl")
{
  die "Error!  setup.pl is not executable!\n";
}

if (length $prefix == 0)
{
  print "Running setup.pl to configure $appName in the Portal...  Prefix = '$prefix'\n";

  system("$location/scripts/setup.pl");

  # at this point Portal::App::Data::Config should be usable.
}

if (-e "$location/scripts/appinstall.pl")
{ # if doing an rpm build, we don't register the app with the portal!
  print "\nDo you want to register the App with the Portal? [Y]: " if (length $prefix == 0);
  my $input = (length $prefix == 0 ? "" : "N");
  $input = getInput($input, "y");
  if ($input =~ /^y$/i)
  {
    system("$location/scripts/appinstall.pl") == 0 or die "Error running appinstall.pl!  Error = $!\n";
  }
  else
  {
    print "\nSkipping App registration...\n";
  }
}

if (-e "$location/scripts/colorSchemeInstall.pl")
{ # if doing an rpm build, we don't install colorSchemes into the Portal!
  print "\nDo you want to install your colorSchemes? [Y]: " if (length $prefix == 0);
  my $input = (length $prefix == 0 ? "" : "N");
  $input = getInput($input, "y");
  if ($input =~ /^y$/i)
  {
    system("$location/scripts/colorSchemeInstall.pl --appname $appName") == 0 or die "Error running colorSchemeInstall.pl!  Error = $!\n";
  }
  else
  {
    print "\nSkipping colorScheme installation...\n";
  }
}

# now put the configuration scripts into /usr/lib/pcx_portal/$appName.
if (! -d "$prefix/usr/lib/pcx_portal/$appName")
{
  # create the directory.
  print "Creating '$prefix/usr/lib/pcx_portal/$appName'...\n";
  system("/bin/mkdir -p -m 0755 $prefix/usr/lib/pcx_portal/$appName") == 0 or die "Error:  Couldn't create '$prefix/usr/lib/pcx_portal/$appName' directory!  $!\n";
}
else
{
  print "\n'$prefix/usr/lib/pcx_portal/$appName' already exists.  Skipping creation...\n";
  # remove old files
  system("/bin/rm -Rf $prefix/usr/lib/pcx_portal/$appName/*") == 0 or die "Error removing old files from '$prefix/usr/lib/pcx_portal/$appName'!  Error = $!\n";
}
my @files = glob("$location/scripts/*.pl");
print "Copying support scripts to '$prefix/usr/lib/pcx_portal/$appName/'...\n";
foreach my $file (@files)
{
  system("/bin/cp -f $file $prefix/usr/lib/pcx_portal/$appName/") == 0 or die "Error copying '$file' to '$prefix/usr/lib/pcx_portal/$appName'!  Error = $!\n";
}

print "\ninstall_app.pl execution complete...\n\n";

# string getInput(string, default)
# pass in any initial values you want or an empty string and the default value to use
# if the user presses Enter.  You get back what was either passed in or what the user entered.
sub getInput
{
  my $input = shift;
  my $default = shift;
  while (length $input == 0)
  {
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $default;
    }
  }
  return $input;
}
