#! /usr/bin/perl
# setup.pl - This script is run to configure the Portal.
# Created by James A. Pattie, 06/30/2001.  Copyright (c) 2000-2003 Xperience, Inc. http://www.pcxperience.com/
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.

use strict;
use Portal::Language;
use Portal::XML::ConfigParser;
use Portal::XML::ConfigObject;
use Portal::Data::DetectDistro;

$| = 1;

my $debug = 0;  # display any of the debug info (off by default).
my $configVersion = "0.6";  # Current Configuration File Version to be working with.

# check the /etc/issue file to see what distro we are using and go accordingly
my @distroInfo = Portal::Data::DetectDistro::detect();
if (!defined @distroInfo)
{
  die "Error: " . $Portal::Data::DetectDistro::errorMessage;
}
my $distroName = $distroInfo[0];
my $distroVersion = $distroInfo[1];

print "Detected Distro '$distroName', Version '$distroVersion'\n\n" if ($debug);

print "\n#################  Portal Configuration settings #################\n";

# get the prefix value and validate it, if specified.
my $prefix = "";
if ($#ARGV == 0)
{
  $prefix = $ARGV[0];
  if (length $prefix && not -d $prefix)
  {
    die "Error!  Prefix directory = '$prefix' does not exist!\n";
  }
}

my $configFile = $prefix . "/etc/pcx_portal/Portal.xml";

# make sure the config file exists.
if (! -e $configFile)
{
  die "Error:  Config file = '$configFile' does not exist!\n";
}

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = Portal::XML::ConfigParser->new(langObj => $langObj);
my $settings = undef;
my $done = 0;
my $version = $configVersion;

# pull in the settings from the config file.
while (!$done)
{
  eval { $settings = $configObj->parse(file => $configFile, module => "Portal", version => $version); };
  if ($@)
  {
    if ($@ =~ /(  '(\d+\.\d+)' is not equal to Version '$version'!)/m)
    {
      $version = $2;  # set the version to what the config file is.
      print "Downgrading to version '$version'\n" if ($debug);
    }
    else
    {
      die $@;
    }
  }
  else
  {
    $done = 1;  # we are done.
  }
}

RESTART:  # We go here if they say No to use current settings.
my $useCurrent = 0;
my $input = "";
my $configItems = Portal::XML::ConfigObject->new(langObj => $langObj);
if ($configItems->error)
{
  die $configItems->errorMessage;
}

# Make sure we get all values that were there before so that they get put back out if we don't modify them.
$configItems->version($settings->version());
print "version = '$configItems->{version}'\n" if ($debug);
$configItems->module($settings->module());
print "module = '$configItems->{module}'\n" if ($debug);
foreach my $setting (keys %{$settings->settings()})
{
  # fixup any true/false -> 1/0 conversions.
  if ($setting =~ /^(helpDefined|cookieDebug)$/)
  {
    if ($settings->settings->{$setting} eq "true")
    {
      $settings->settings->{$setting} = 1;
    }
    if ($settings->settings->{$setting} eq "false")
    {
      $settings->settings->{$setting} = 0;
    }
  }
  $configItems->settings->{$setting} = $settings->settings->{$setting};
  print "$setting = '$configItems->{settings}->{$setting}'\n" if ($debug);
}

# take care of any new items for this version of the config file.
if ($settings->{version} eq "0.1")
{
  $settings->{settings}->{hostingURL} = "HOSTINGURL";
  $settings->{settings}->{hostingAltTag} = "HOSTINGALTTAG";
  $settings->{settings}->{hostingLogoUrl} = "HOSTINGLOGOURL";

  # make sure the static color entries make it into the new config file.
  $configItems->{settings}->{loginbgColor} = "#E6E6FF";
  $configItems->{settings}->{loginfgColor} = "black";
  $configItems->{settings}->{bgColor} = "white";
  $configItems->{settings}->{fgColor} = "black";
  $configItems->{settings}->{errorfgColor} = "red";
  $configItems->{settings}->{errorbgColor} = "lightblue";
  $configItems->{settings}->{menubgColor} = "#0091AA";
  $configItems->{settings}->{menufgColor} = "white";
  $configItems->{settings}->{cellfgColor} = "black";
  $configItems->{settings}->{cellbgColor} = "lightblue";
  $configItems->{settings}->{cellTitlefgColor} = "white";
  $configItems->{settings}->{cellTitlebgColor} = "darkblue";
  $configItems->{version} = "0.2";
}
if ($configItems->{version} eq "0.2")
{
  $configItems->{settings}->{helpDefined} = 1;
  $configItems->{version} = "0.3";
}
if ($configItems->{version} eq "0.3")
{
  $configItems->{settings}->{cookieDebug} = "0";
  $settings->{settings}->{cookieDebug} = "0";
  $configItems->{version} = "0.4";
}
if ($configItems->{version} eq "0.4")
{
  $settings->{settings}->{webUser} = "WEBUSER";
  $settings->{settings}->{webGroup} = "WEBGROUP";
  $configItems->{version} = "0.5";
}
if ($configItems->{version} eq "0.5")
{
  $settings->{settings}->{dbAdminUser} = "DBADMINUSER";
  $settings->{settings}->{dbAdminPasswd} = "DBADMINPASSWD";

  # remove all the Color entries from the config file.
  foreach my $color (qw(loginbgColor loginfgColor bgColor fgColor errorbgColor errorfgColor menubgColor menufgColor cellbgColor cellfgColor cellTitlebgColor cellTitlefgColor))
  {
    delete $settings->{settings}->{$color};
  }

  $configItems->{version} = "0.6";
}

if (length $prefix == 0)
{
  # get the HostName of the Portal machine.
  my $temp = `hostname`; chomp $temp;
  $configItems->{settings}->{myHostName} = getInput(prompt => "Enter Host Name of Portal machine", value => ($settings->{settings}->{myHostName} ne "MYHOSTNAME" ? $settings->{settings}->{myHostName} : $temp), mask => "/^(.+\\.\\D+|\\d+\\.\\d+\\.\\d+\\.\\d+)\$/");

  # get the Web Server Type (http or https)
  $configItems->{settings}->{httpType} = getInput(prompt => "Enter Web Server Type (http or https)", value => ($settings->{settings}->{httpType} ne "HTTPTYPE" ? $settings->{settings}->{httpType} : "https"), mask => "/^(http|https)\$/");
}

# get the Session Type
$input = getInput(prompt => "Enter Session Type (D)atabase, (F)ile", value => ($settings->{settings}->{sessionType} ne "SESSIONTYPE" ? ($settings->{settings}->{sessionType} eq "Database" ? "D" : "F") : "D"), mask => "/^(D|F|d|f)\$/");
if ($input =~ /^d$/i)
{
  $input = "Database";
}
elsif ($input =~ /^f$/i)
{
  $input = "File";
}
$configItems->{settings}->{sessionType} = $input;

if ($configItems->{settings}->{sessionType} eq "File")
{
  my $sessionDirectory = "";
  my $sessionLockDir = "";

  $sessionDirectory = getInput(prompt => "Enter Session Directory", value => ($settings->{settings}->{sessionDirectory} ne "SESSIONDIRECTORY" ? $settings->{settings}->{sessionDirectory} : "/var/spool/portal/sessions"));

  $sessionLockDir = getInput(prompt => "Enter Session Lock Directory", value => ($settings->{settings}->{sessionLockDir} ne "SESSIONLOCKDIR" ? $settings->{settings}->{sessionLockDir} : $sessionDirectory . "/lock"));

  $configItems->{settings}->{sessionDirectory} = $sessionDirectory;
  $configItems->{settings}->{sessionLockDir} = $sessionLockDir;
}

if (length $prefix == 0)
{
  # get the Database Type
  $input = getInput(prompt => "Enter Database Type (P)ostgreSQL, (M)ySQL", value => ($settings->{settings}->{dbType} ne "DBTYPE" ? ($settings->{settings}->{dbType} eq "Postgres" ? "P" : "M") : "P"), mask => "/^(P|M)\$/i");
  if ($input =~ /^p$/i)
  {
    $input = "Postgres";
  }
  elsif ($input =~ /^m$/i)
  {
    $input = "MySQL";
  }
  $configItems->{settings}->{dbType} = $input;

  my $dbLocal = getInput(prompt => "Are the databases on this machine? (Y/n)", value => "Y", mask => "/^[yn]\$/i");
  $dbLocal = ($dbLocal =~ /^y$/i ? 1 : 0);

  # get the Database Server Host
  $configItems->{settings}->{dbHost} = getInput(prompt => "Enter Portal Database Server Host Name or IP Address", value => ($settings->{settings}->{dbHost} ne "DBHOST" ? $settings->{settings}->{dbHost} : ($dbLocal ? "127.0.0.1" : $configItems->{settings}->{myHostName})));

  # get the Database Port
  $configItems->{settings}->{dbPort} = getInput(prompt => "Enter Database Server Port", value => ($settings->{settings}->{dbPort} ne "DBPORT" ? $settings->{settings}->{dbPort} : ($configItems->{settings}->{dbType} eq "Postgres" ? "5432" : "3306")), mask => "/^(\\d+)\$/");

  # get the Database Admin User
  $configItems->{settings}->{dbAdminUser} = getInput(prompt => "Enter Database Administrative User Name", value => ($settings->{settings}->{dbAdminUser} ne "DBADMINUSER" ? $settings->{settings}->{dbAdminUser} : ($configItems->{settings}->{dbType} eq "Postgres" ? "postgres" : "root")), mask => "/^(\\w+)\$/");

  # get the Database Admin Password
  $configItems->{settings}->{dbAdminPasswd} = getInput(prompt => "Enter Database Administrative User Password", value => ($settings->{settings}->{dbAdminPasswd} ne "DBADMINPASSWD" ? $settings->{settings}->{dbAdminPasswd} : ""), mask => "/^(.+)\$/");

  # get the Database Name
  $configItems->{settings}->{dbName} = getInput(prompt => "Enter Database Name", value => ($settings->{settings}->{dbName} ne "DBNAME" ? $settings->{settings}->{dbName} : "portal_db"), mask => "/^(.+)\$/");

  # get the Billing Database Server Host
  $configItems->{settings}->{billingdbHost} = getInput(prompt => "Enter Billing Database Server Host Name or IP Address", value => ($settings->{settings}->{billingdbHost} ne "BILLINGDBHOST" ? $settings->{settings}->{billingdbHost} : ($dbLocal ? "127.0.0.1" : $configItems->{settings}->{dbHost})));

  # get the Billing Database Name
  $configItems->{settings}->{billingdbName} = getInput(prompt => "Enter Billing Database Name", value => ($settings->{settings}->{billingdbName} ne "BILLINGDBNAME" ? $settings->{settings}->{billingdbName} : "billing_db"));

  # get the Database User Name
  $configItems->{settings}->{dbUser} = getInput(prompt => "Enter Database User Name", value => ($settings->{settings}->{dbUser} ne "DBUSER" ? $settings->{settings}->{dbUser} : "portal"));

  # get the Database User Password
  $configItems->{settings}->{dbPasswd} = getInput(prompt => "Enter Database User Password", value => ($settings->{settings}->{dbPasswd} ne "DBPASSWD" ? $settings->{settings}->{dbPasswd} : "", mask => "/^(.+)\$/"));
}

# get the root directory of the web server
$input = "";
my %dirs = ( 0 => "/usr/share/pcx_portal\t\t- PCXPortal Default", # Portal Default now
	     1 => "/home/httpd/html\t\t\t- Red Hat < 7.0",
             2 => "/home/httpd/htdocs",
             3 => "/home/httpd/docs",
             4 => "/var/www/html\t\t\t- Red Hat >= 7.0",
	     5 => "/var/www\t\t\t\t- Debian",  # debian
             6 => "Other" );

my %revDirs = ( "/home/httpd/html" => 1,
                "/home/httpd/htdocs" => 2,
                "/home/httpd/docs" => 3,
                "/var/www/html" => 4,
		"/var/www" => 5,
		"/usr/share/pcx_portal" => 0 );

my %foundDirs = ();

print "\n";
my $defaultLocation = ($settings->{settings}->{webRoot} ne "WEBROOT" ? (exists $revDirs{$settings->{settings}->{webRoot}} ? $revDirs{$settings->{settings}->{webRoot}} : 6) : 0);
foreach my $op (sort { $a <=> $b } keys %dirs)
{
  (my $dir = $dirs{$op}) =~ s/(\t+.+)$//;  # remove the comments.
  my $found = ($op != 6 && -d "$prefix$dir" ? 1 : 0);
  $foundDirs{$op} = $dir if ($found);
  print $op . ") " . $dirs{$op} . "\n";
}
if ($defaultLocation == 0)
{
  if ($distroName eq "Red Hat Linux")
  {
    if ($foundDirs{4})
    {
      $defaultLocation = 4;
    }
    elsif ($foundDirs{1})
    {
      $defaultLocation = 1;
    }
  }
}
while (length $input == 0)
{
  print "\nSelect the root directory of your web server (0-6)" . ($defaultLocation != -1 ? " [$defaultLocation] " : "") . ": ";
  $input = <STDIN>;
  chomp $input;
  $input = $defaultLocation if (length $input == 0);
  if (exists $dirs{$input})
  {
    if ($input eq "6")
    {
      $input = "";
      print "\nEnter the root directory of your web server";
      print " [$settings->{settings}->{webRoot}]" if ($settings->{settings}->{webRoot} ne "WEBROOT");
      print ": ";
      $input = <STDIN>;
      chomp $input;
      if (length $input > 0)
      {
        if (! -d "$prefix$input")
        {
          print "'$input' does not exist!\n";
          $input = "";
        }
      }
      else
      {
        if ($settings->{settings}->{webRoot} ne "WEBROOT")
        {
          $input = $settings->{settings}->{webRoot};
          if (! -d "$prefix$input")
          {
            print "'$input' does not exist!\n";
            $input = "";
          }
        }
      }
    }
    else
    {
      $input = $dirs{$input};  # assign the selected choice.
      $input =~ s/(\t+.+)$//;  # remove the comments.
      if (! -d "$prefix$input")
      {
        print "'$input' does not exist!\n";
        $input = "";
      }
    }
  }
}

if ($input =~ /.+\/$/)
{ # we need to strip off the trailing /.
  $input =~ s/(.+)\/$/$1/;
}
$configItems->{settings}->{webRoot} = $input;

# get the name of the directory to install in in the web site.
$configItems->{settings}->{siteDir} = getInput(prompt => "Enter directory to install in", value => ($settings->{settings}->{siteDir} ne "SITEDIR" ? $settings->{settings}->{siteDir} : "/portal"), mask => "/^(\\/[^ ]+)\$/");

# figure out the webUser and webGroup values.
my $defaultUser = "nobody";
if ($distroName =~ /Debian GNU/)
{
  $defaultUser = "www-data";
  print "Setting defaultUser = $defaultUser...\n" if ($debug);
}
if ($distroName =~ /Red Hat Linux/ && $distroVersion >= 7.0)
{
  $defaultUser = "apache";
  print "Setting defaultUser = $defaultUser...\n" if ($debug);
}
# need to check other distros as we learn about them. (Mandrake, etc.)

my $defaultGroup = $defaultUser;

my $valid = 0;
while (!$valid)
{
  $configItems->{settings}->{webUser} = getInput(prompt => "What is the user the webserver runs as", value => ($settings->{settings}->{webUser} ne "WEBUSER" ? $settings->{settings}->{webUser} : $defaultUser), mask => "/^(.+)\$/");
  my $found = `/bin/cat /etc/passwd | /bin/egrep ^$configItems->{settings}->{webUser}: | /usr/bin/wc -l`;
  if ($found == 1)
  {
    $valid = 1;
  }
  else
  {
    print "User = '$configItems->{settings}->{webUser}' does not exist in /etc/passwd!\n";
    $configItems->{settings}->{webUser} = $settings->{settings}->{webUser};
  }
}
$valid = 0;
while (!$valid)
{
  $configItems->{settings}->{webGroup} = getInput(prompt => "What is the group the webserver runs as", value => ($settings->{settings}->{webGroup} ne "WEBGROUP" ? $settings->{settings}->{webGroup} : $defaultGroup), mask => "/^(.+)\$/");
  my $found = `/bin/cat /etc/group | /bin/egrep ^$configItems->{settings}->{webGroup}: | /usr/bin/wc -l`;
  if ($found == 1)
  {
    $valid = 1;
  }
  else
  {
    print "Group = '$configItems->{settings}->{webGroup}' does not exist in /etc/group!\n";
    $configItems->{settings}->{webGroup} = $settings->{settings}->{webGroup};
  }
}

# Find out if the user wants to display cookieDebug info.
$configItems->{settings}->{cookieDebug} = getInput(prompt => "Display cookie Debug info (y/N)", value => ($settings->{settings}->{cookieDebug} ne "COOKIEDEBUG" ? ($settings->{settings}->{cookieDebug} ? "Y" : "N") : "N"), mask => "/^([YN])\$/i");
  $configItems->{settings}->{cookieDebug} = ($configItems->{settings}->{cookieDebug} =~ /^Y$/i ? 1 : 0);

if (length $prefix == 0)
{
  # get the cookie Domain value.  Try to use the myHostName value first.
  my $domain = $configItems->{settings}->{myHostName};
  $configItems->{settings}->{cookieDomain} = getInput(prompt => "Enter your cookie Domain (must start with a ., only if you want a complete domain, and contain 2 periods or can be an IP Address)", value => ($settings->{settings}->{cookieDomain} ne "COOKIEDOMAIN" ? $settings->{settings}->{cookieDomain} : $domain), mask => "/^(\\.?\\w+\\.\\D+|\\d+\\.\\d+\\.\\d+\\.\\d+)\$/");

  # gather cookie Life value
  my $cookieLife = ($settings->{settings}->{cookieLife} ne "COOKIELIFE" ? $settings->{settings}->{cookieLife} : "30");
  $input = "";
  while (length $input == 0)
  {
    print "\nEnter Life time (in minutes) of session cookies [$cookieLife]: ";
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $cookieLife;
    }
    elsif ($input < 30)
    {
      print "'$input' is invalid!  Minimum time is 30 minutes.\n";
      $input = "";
    }
  }
  $configItems->{settings}->{cookieLife} = $input;

  # gather the email address to use for JavaScript Errors.
  $domain = $configItems->{settings}->{cookieDomain};
  $domain =~ s/^(\.)(.+)$/$2/;  # strip off the leading .
  $domain = "portal\@" . $domain;
  $configItems->{settings}->{emailAddress} = getInput(prompt => "Enter your JavaScript Error Handler email address", value => ($settings->{settings}->{emailAddress} ne "EMAILADDRESS" ? $settings->{settings}->{emailAddress} : $domain), mask => "/^(.+@([\\w-]+\\D+|\\d+\\.\\d+\\.\\d+\\.\\d+))\$/");

  # see if they want to enable file uploads.
  $configItems->{settings}->{allowFileUploads} = getInput(prompt => "Allow File Uploads", value => ($settings->{settings}->{allowFileUploads} ne "ALLOWFILEUPLOADS" ? ($settings->{settings}->{allowFileUploads} ? "Y" : "N") : "Y"), mask => "/^([YN])\$/i");
  $configItems->{settings}->{allowFileUploads} = ($configItems->{settings}->{allowFileUploads} =~ /^Y$/i ? 1 : 0);

  # gather the other info for File Uploads.
  if ($configItems->{settings}->{allowFileUploads})
  {
    my $maxFileUploadSize = "";
    my $fileUploadLocation = "";
    my $fileUploadPrefix = "";

    # gather the Max Input Stream Size value.
    $maxFileUploadSize = ($settings->{settings}->{maxFileUploadSize} ne "MAXFILEUPLOADSIZE" ? $settings->{settings}->{maxFileUploadSize} : "4M");
    if ($maxFileUploadSize !~ /^(\d+)(M|K|B)$/)
    {
      # convert to Human readable version.
      my $tempH = ($maxFileUploadSize / (1024 * 1024));  # try for M first.
      if ($tempH < 0)
      {
        $tempH = ($maxFileUploadSize / 1024);  # try for K next.
        $maxFileUploadSize = $tempH . "K";
      }
      else
      {
        $maxFileUploadSize = $tempH . "M";
      }
    }
    $input = "";
    while (length $input == 0)
    {
      print "\nMax File Upload Size in (B)ytes, (M)egs or (K)iloBytes [$maxFileUploadSize]: ";
      $input = <STDIN>;
      chomp $input;
      if (length $input == 0)
      {
        $input = $maxFileUploadSize;
      }
      if ($input !~ /^\d+(M|K|B)?$/)
      {
        print "'$input' is invalid!\n";
        $input = "";
      }
      else
      {
        # fixup $input
        (my $tempI = $input) =~ s/^(\d+)(M|K|B)$/($2 eq "M" ? $1 * 1024 * 1024 : ($2 eq "K" ? $1 * 1024 : $1))/e;
        if ($tempI < 131072)
        {
          print "'$input' is less than 128K = 131072 bytes!\n";
          $input = "";
        }
      }
    }
    $configItems->{settings}->{maxFileUploadSize} = $input;

    # gather the location to write files to (/tmp).
    $fileUploadLocation = ($settings->{settings}->{fileUploadLocation} ne "FILEUPLOADLOCATION" ? $settings->{settings}->{fileUploadLocation} : "/tmp");
    $input = "";
    while (length $input == 0)
    {
      print "\nFile Upload Location [$fileUploadLocation]: ";
      $input = <STDIN>;
      chomp $input;
      if (length $input == 0)
      {
        $input = $fileUploadLocation;
      }
      if (! -d $input)
      {
        print "'$input' is invalid!  Does not exist or is not a directory.\n";
        $input = "";
      }
    }
    $configItems->{settings}->{fileUploadLocation} = $input;

    # gather the name to prefix to the files (cgi-lib).
    $configItems->{settings}->{fileUploadPrefix} = getInput(prompt => "File Upload Prefix name", value => ($settings->{settings}->{fileUploadPrefix} ne "FILEUPLOADPREFIX" ? $settings->{settings}->{fileUploadPrefix} : "cgi-lib"), mask => "/^([^\\d\\$\\%\\\\-][\\d\\w\\s-_\\.]+)\$/");
  }

  # get the hosting companies URL, ALT String and Logo Url
  $configItems->{settings}->{hostingURL} = getInput(prompt => "Your website url", value => ($settings->{settings}->{hostingURL} ne "HOSTINGURL" ? $settings->{settings}->{hostingURL} : "http://" . $configItems->{settings}->{myHostName}), mask => "/^(https?:\\/\\/[^\\/]+\\.[^\\/]+.*)\$/");
  $configItems->{settings}->{hostingAltTag} = getInput(prompt => "Your Alt Tag value (company name, etc.)", value => ($settings->{settings}->{hostingAltTag} ne "HOSTINGALTTAG" ? $settings->{settings}->{hostingAltTag} : ""), mask => "/^(.+)\$/");
  (my $hostingURL = $configItems->{settings}->{hostingURL}) =~ s/^(https?:\/\/[^\/]+\.[^\/]+)(.*)$/$1/;
  $configItems->{settings}->{hostingLogoUrl} = getInput(prompt => "Your logo's url", value => ($settings->{settings}->{hostingLogoUrl} ne "HOSTINGLOGOURL" ? $settings->{settings}->{hostingLogoUrl} : $hostingURL . "/images/logo.png"), mask => "/^(https?:\\/\\/[^\\/]+\\.[^\\/]+\\/.+\\.(gif|png|jpg|tif|bmp))\$/");
}
# verify they want to use these settings.
$input = getInput(prompt => "Accept these values (Y) or Redo (N)?", value => "Y", mask => "/^([YN])\$/i");
if ($input =~ /^N$/i)
{
  goto RESTART;
}

# validate the configuration object.
if (!$configItems->isValid())
{
  die "Portal settings not valid!\n\n" . $configItems->errorMessage . "\n";
}

print "Generating Config file '$configFile'...\n";
print $configItems->generateXML if ($debug);  # debug output.

# generate the Config file now.
open (F, ">$configFile") or die "Error:  Couldn't open '$configFile'!  Error = $!\n";

print F $configItems->generateXML;

close (F)		or die "can't close $configFile: $!";

# getInput - prompt, mask, and value
# mask is the string representing the regular expression to execute including any operator needed.
sub getInput
{
  my %args = (prompt => "", mask => "/^(.*)\$/", value => "", @_);
  my $prompt = $args{prompt};
  my $mask = $args{mask};
  my $value = $args{value};

  my $input = "";
  my $done = 0;
  while (!$done)
  {
    print "\n$prompt [$value]: ";
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $value;
    }
    my $result;
    eval("\$result = \$input !~ $mask");
    if ($@)
    {
      die "Invalid eval on mask '$mask', input '$input', result '$result'\n" . $@ . "\n";
    }
    elsif ($result == 1)
    {
      print "'$input' is invalid!\n";
      $input = "";
    }
    else
    {
      $done = 1;  # This will let me out even if I'm entering an empty string, as long as that is valid.
    }
  }
  $input =~ s/@/\\@/g;  # make sure that any @'s are escaped.
  $input =~ s/%/\\%/g;  # make sure that any %'s are escaped.

  return $input;
}

1;
