#! /usr/bin/perl
# This script will make the requested database changes using the API's defined
# for working with the config, dynamic_content and rights tables.
# The event_tb will need methods defined, if I want to make it more easily
# accessible other than via this script.

# updateDBSettings.pl - Created by James Pattie, (james@pcxperience.com)
# Copyright (c) 2002-2005, Xperience, Inc. (http://www.pcxperience.com/)
# All rights reserved.  This program is free software; you can redistribute it
# and/or modify it under the same terms as Perl itself.
# 2005-03-01

use Getopt::Long;
use Portal::Language;
use Portal::Data::Config;
use Portal::Auth;
use Portal::Log;
use Portal::Application;
use Portal::Data::Variables;
use Portal::Methods;
use Portal::XML::DBSettingsParser;
use strict;

# output a header so they at least have a clue what is going on in the
# postinst steps of the Debian packages.
print "updateDBSettings: ";

my $syntax = "updateDBSettings.pl --appname appName [--file file]\n";
my $configVersion = "1.0";  # version of dbSettings XML config file.

# get the command line info so we know what application is trying to update the Portal database.
my $appName = "";
my $fileName = "";
GetOptions('appname=s' => \$appName, 'file=s' => \$fileName);

if (length $appName == 0)
{
  die "Error:  You did not specify the appname to use!\n$syntax";
}

# instantiate with language 'en' - English
my $langObj = Portal::Language->new(lang => 'en');
if ($langObj->error)
{
  die "Error:  Instantiating the Language Object failed!\n" . $langObj->errorMessage;
}

my $configObj = undef;
eval "\$configObj = Portal::Data::Config->new(langObj => \$langObj);";
if ($@)
{
  die "Error instantiating Portal::Data::Config->new()!  Error = $@\n";
}
if ($configObj->error)
{
  die $configObj->errorMessage;
}

my $methods = Portal::Methods->new(langObj => $langObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}

# make connection to the portal database
my $portalDB = $methods->portalDBSetup(type => "portal", configObj => $configObj);
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($portalDB->error)
{
  myDie(error => $portalDB->errorMessage, configObj => $configObj);
}

# setup a sessionObj to work with.
my $session = $methods->portalSession(portalDB => $portalDB, sessionId => "",
                                              configObj => $configObj, mode => "write");
if ($methods->error)
{
  myDie(error => $methods->errorMessage, configObj => $configObj);
}
if ($session->error)
{
  myDie(error => $session->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Application module.
my $applicationObj = Portal::Application->new(portalDB => $portalDB, sessionObj => $session, langObj => $langObj);
if ($applicationObj->error)
{
  myDie(error => $applicationObj->errorMessage, configObj => $configObj);
}

# instantiate an instance of the Auth module.
my $authObj = Portal::Auth->new(portalDB => $portalDB, sessionObj => $session, langObj => $langObj);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}

# ask for the system admin account to use.

ADMIN:
print "What system administrator account should I use [admin]: ";
my $admin = getInput("", "admin");

# validate the username specified.
my $userObj = $authObj->getUserInfo(uname => $admin);
if ($authObj->error)
{
  myDie(error => $authObj->errorMessage, configObj => $configObj);
}
if (defined $userObj)
{
  if ($userObj->error)
  {
    myDie(error => $userObj->errorMessage, configObj => $configObj);
  }
}
else
{
  print "User '$admin' not found!  Retry...\n";
  goto ADMIN;
}

# now verify they are a system administrator
if (!$userObj->{sysadmin})
{
  print "User '$admin' is not a system administrator!  Retry...\n";
  goto ADMIN;
}
$session->{store}->{userObj} = $userObj;  # store the userObj in the session.
$session->{store}->{changed} = 1;

print "file = '$fileName'\n";
if (length $fileName == 0)
{
  $fileName = $configObj->webRoot . $configObj->siteDir . "/install/$appName/DBSettings.xml";
}

if (!-e $fileName)
{
  print "No DBSettings.xml config file to process!  Skipping...\n\n";
  exit 0;
}

print "Processing $fileName...\n";
my $file = $fileName;
# parse the dbSettings xml file.
my $cfgObj = undef;
eval { $cfgObj = Portal::XML::DBSettingsParser->new(langObj => $langObj); };
if ($@)
{
  myDie(error => "Instantiation of DBSettingsParser failed!<br>\n$@", configObj => $configObj);
}
my $settingsObj = undef;
eval { $settingsObj = $cfgObj->parse(file => $file, module => "$appName", version => $configVersion); };
if ($@)
{
  myDie(error => "Parse of '$file' failed!<br>\n$@", configObj => $configObj);
}

# we now have a seemingly valid dbSettings object.  work with it.
foreach my $section (keys %{$settingsObj->{sections}})
{
  # process the groups in the following order: add, update, delete
  foreach my $group (qw(add update delete))
  {
    if (exists $settingsObj->{sections}->{$section}->{$group})
    {
      my @entries = @{$settingsObj->{sections}->{$section}->{$group}};
      print "\nProcessing $section $group...\n";
      if ($section eq "config")
      {
        if ($group eq "add")
        {
          foreach my $entry (@entries)
          {
            # see if the config entry is already defined.  If not, we create it.
            my $result = $methods->getConfigValue(name => $entry->{name}, portalDB => $portalDB);
            if ($methods->error)
            {
              myDie(error => $methods->errorMessage, configObj => $configObj);
            }
            if (!defined $result)
            {
              print "Creating $section, name = '$entry->{name}', value = '$entry->{value}': ";
              $result = $methods->setConfigValue(name => $entry->{name}, value => $entry->{value}, sessionObj => $session, portalDB => $portalDB);
              if ($methods->error)
              {
                myDie(error => $methods->errorMessage, configObj => $configObj);
              }
              if ($result == -1)
              {
                myDie(error => "name = '$entry->{name}' already exists!", configObj => $configObj);
              }
              elsif ($result == -2)
              {
                myDie(error => "permission denied!", configObj => $configObj);
              }
              elsif ($result == 1)
              {
                print "created\n";
              }
            }
          }
        }
        elsif ($group eq "update")
        {
          foreach my $entry (@entries)
          {
            # see if the config entry is already defined.  If yes and the value is the same, then we update to newValue.
            my $result = $methods->getConfigValue(name => $entry->{name}, portalDB => $portalDB);
            if ($methods->error)
            {
              myDie(error => $methods->errorMessage, configObj => $configObj);
            }
            if ($result eq $entry->{value})
            {
              print "Updating $section, name = '$entry->{name}', value = '$entry->{value}' to '$entry->{newValue}': ";
              $result = $methods->updateConfigValue(name => $entry->{name}, value => $entry->{newValue}, sessionObj => $session, portalDB => $portalDB);
              if ($methods->error)
              {
                myDie(error => $methods->errorMessage, configObj => $configObj);
              }
              if ($result == -1)
              {
                myDie(error => "name = '$entry->{name}' does not exist!", configObj => $configObj);
              }
              elsif ($result == -2)
              {
                myDie(error => "permission denied!", configObj => $configObj);
              }
              elsif ($result == 1)
              {
                print "updated\n";
              }
            }
          }
        }
        elsif ($group eq "delete")
        {
          foreach my $entry (@entries)
          {
            # see if the config entry is defined.  If yes, then we delete it.
            my $result = $methods->getConfigValue(name => $entry->{name}, portalDB => $portalDB);
            if ($methods->error)
            {
              myDie(error => $methods->errorMessage, configObj => $configObj);
            }
            if (defined $result)
            {
              print "Deleting $section, name = '$entry->{name}': ";
              $result = $methods->deleteConfigValue(name => $entry->{name}, sessionObj => $session, portalDB => $portalDB);
              if ($methods->error)
              {
                myDie(error => $methods->errorMessage, configObj => $configObj);
              }
              if ($result == -1)
              {
                myDie(error => "name = '$entry->{name}' does not exist!", configObj => $configObj);
              }
              elsif ($result == -2)
              {
                myDie(error => "permission denied!", configObj => $configObj);
              }
              elsif ($result == 1)
              {
                print "deleted\n";
              }
            }
          }
        }
      }
      elsif ($section eq "event")
      {
        if ($group eq "add")
        {
        }
        elsif ($group eq "update")
        {
        }
        elsif ($group eq "delete")
        {
        }
      }
      elsif ($section eq "rights")
      {
        if ($group eq "add")
        {
        }
        elsif ($group eq "update")
        {
        }
        elsif ($group eq "delete")
        {
        }
      }
      elsif ($section eq "dynamic-content")
      {
        if ($group eq "add")
        {
        }
        elsif ($group eq "update")
        {
        }
        elsif ($group eq "delete")
        {
        }
      }
    }
  }
}

# cleanup the session.
$session->delete;

print "\nupdateDBSettings.pl execution complete...\n";
exit 0;

# myDie - Takes lang, encoding, error, configObj
sub myDie
{
  my %args = ( lang => 'en', encoding => 'iso-8859-1', error => "", configObj => undef, @_ );
  my $lang = $args{lang};
  my $encoding = $args{encoding};
  my $message = $args{error};
  my $configObj = $args{configObj};
  my $dateStamp = $methods->getCurrentDate(format => "%F %T");

  print("updateDBSettings: Error Occurred!\n");
  print($message);
  print("Have the Administrator check the error log ($dateStamp).\n");
  if (defined $portalDB)
  {
    my $logObj = Portal::Log->new(dbHandle => $portalDB, langObj => $langObj);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
    my $hostname = `hostname -i`;
    chomp $hostname;
    $hostname =~ s/ //g;

    my $logEntry = Portal::Objects::LogEntry->new(action => 18, ipAddress => $hostname, extraInfo => $message, userId => 0, langObj => $langObj);
    if ($logEntry->error)
    {
      die $logEntry->errorMessage;
    }
    $logObj->newEntry(logEntry => $logEntry);
    if ($logObj->error)
    {
      die $logObj->errorMessage;
    }
  }
  exit 1;
}

# string getInput(string, default)
# pass in any initial values you want or an empty string and the default value to use
# if the user presses Enter.  You get back what was either passed in or what the user entered.
sub getInput
{
  my $input = shift;
  my $default = shift;
  while (length $input == 0)
  {
    $input = <STDIN>;
    chomp $input;
    if (length $input == 0)
    {
      $input = $default;
    }
  }
  return $input;
}

